<?php

namespace Albedo\ScratchCard\Models;

use Albedo\ScratchCard\Contracts\PrizeContract;
use Albedo\ScratchCard\Enums\ScratchCardPrizeOrderStatus;
use Albedo\ScratchCard\Enums\TypeOfPrizeEnum;
use App\Models\User;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Routing\Exceptions\BackedEnumCaseNotFoundException;

class ScratchCardEntry extends Model implements PrizeContract
{
    protected $guarded = [];

    protected $casts = [
        'drawn_at' => 'datetime',
        'scratched' => 'boolean',
        'status' => ScratchCardPrizeOrderStatus::class,
    ];

    public function scratchCardPrizeDraw(): BelongsTo
    {
        return $this->belongsTo(ScratchCardPrizeDraw::class);
    }

    public function scratchCardPrizeOrder(): HasOne
    {
        return $this->hasOne(ScratchCardPrizeOrder::class)
            ->latest();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function getRequiredFormShipping(): bool
    {
        return $this->scratchCardPrizeDraw
            ->getRequiredFormShipping();
    }

    public function getCompletedShipping(): bool
    {
        if (!$this->getRequiredFormShipping()) {
            return false;
        };

        return match ($this->status) {

            ScratchCardPrizeOrderStatus::NO_DATA,
            ScratchCardPrizeOrderStatus::NEEDS_CORRECTION => false,

            ScratchCardPrizeOrderStatus::REJECTED,
            ScratchCardPrizeOrderStatus::IN_VERIFICATION,
            ScratchCardPrizeOrderStatus::APPROVED,
            ScratchCardPrizeOrderStatus::PRIZE_ISSUED, => true,

            default => throw new BackedEnumCaseNotFoundException(ScratchCardPrizeOrderStatus::class, $this->value),
        };
    }

    public function assignWinner(User $user): void
    {
        $this->update([
            'user_id' => $user->getKey(),
            'drawn_at' => now(),
        ]);
    }

    public function prizeName(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->scratchCardPrizeDraw->name
        );
    }
}
