<?php

namespace Albedo\ScratchCard\Models;

use Albedo\ScratchCard\Contracts\PrizeContract;
use Albedo\ScratchCard\Dtos\ScratchCardPrizeDto;
use Albedo\ScratchCard\Enums\ScratchCardTypeEnum;
use Albedo\ScratchCard\Services\ScratchCardPrizeService;
use App\Models\User;
use Carbon\CarbonPeriod;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class Gate extends Model
{
    public const PRIZE_MORPH_NAME = 'prize';

    public $timestamps = false;

    public $guarded = [
        'id'
    ];

    public $casts = [
        'open_gate_at' => 'datetime',
        'close_gate_at' => 'datetime',
        'received' => 'datetime',
        'type' => ScratchCardTypeEnum::class,
    ];

    public function prize(): MorphTo
    {
        return $this->morphTo(self::PRIZE_MORPH_NAME);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public static function draw(ScratchCardTypeEnum $type): null|PrizeContract
    {
        $gate = Gate::query()
            ->where('open_gate_at', '<=', now())
            ->where('close_gate_at', '>=', now())
            ->whereNull('received')
            ->whereType($type)
            ->first();

        if (!$gate) {
            return null;
        }

        $gate->update(['received' => now()]);

        return $gate->prize;
    }

    public static function createGate(
        string        $name,
        CarbonPeriod  $carbonPeriod,
        PrizeContract $prize,
        ScratchCardTypeEnum $type,
    ): Gate
    {
        return Gate::create([
            'prize_id' => $prize->getKey(),
            'prize_type' => $prize->getMorphClass(),
            'user_id' => null,
            'name' => $name,
            'type' => $type,
            'received' => null,
            'open_gate_at' => $carbonPeriod->getStartDate(),
            'close_gate_at' => $carbonPeriod->getEndDate()
        ]);

    }

}
