<?php

namespace Albedo\ScratchCard\Models;

use Albedo\ScratchCard\Contracts\PrizeContract;
use App\Models\User;
use Carbon\CarbonPeriod;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class ScratchCardGate extends Model
{
    public const PRIZE_MORPH_NAME = 'prize';

    public $timestamps = false;

    public $guarded = [
        'id'
    ];

    public $casts = [
        'open_gate_at' => 'datetime',
        'close_gate_at' => 'datetime',
        'received' => 'datetime',
    ];

    public function prize(): MorphTo
    {
        return $this->morphTo(self::PRIZE_MORPH_NAME);
    }

    public function reason(): BelongsTo
    {
        return $this->belongsTo(ScratchCardReason::class, 'scratch_card_reason_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public static function draw(ScratchCardReason $scratchCardReason): null|PrizeContract
    {
        $gate = self::query()
            ->where('open_gate_at', '<=', now())
            ->where('close_gate_at', '>=', now())
            ->whereNull('received')
            ->where('scratch_card_reason_id', $scratchCardReason->id)
            ->first();

        if (!$gate) {
            return null;
        }

        $gate->update(['received' => now()]);

        return $gate->prize;
    }

    public static function createGate(
        string            $name,
        CarbonPeriod      $carbonPeriod,
        PrizeContract     $prize,
        ScratchCardReason $scratchCardReason,
    ): self
    {
        return self::create([
            'prize_id' => $prize->getKey(),
            'prize_type' => $prize->getMorphClass(),
            'user_id' => null,
            'name' => $name,
            'scratch_card_reason_id' => $scratchCardReason->id,
            'received' => null,
            'open_gate_at' => $carbonPeriod->getStartDate(),
            'close_gate_at' => $carbonPeriod->getEndDate()
        ]);

    }

}
