<?php

namespace Albedo\ScratchCard\Services;

use Albedo\ScratchCard\Enums\ScratchCardPrizeOrderStatus;
use Albedo\ScratchCard\Enums\TypeOfPrizeEnum;
use Albedo\ScratchCard\Models\ScratchCardEntry;
use Albedo\ScratchCard\Models\ScratchCardReason;
use Albedo\ScratchCard\Models\ScratchCardResult;
use Albedo\ScratchCard\Models\ScratchCardType;
use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

readonly class UserScratchCardService
{
    public function addScratchCardResult(
        User                $user,
        ScratchCardReason $scratchCardReason,
        Model               $source
    )
    {
        $scratchCardTypeModel = ScratchCardType::query()
            ->where('scratch_card_reason_id', $scratchCardReason->id)
            ->firstOrFail();

        return ScratchCardResult::query()->create([
            'scratch_card_type_id' => $scratchCardTypeModel->id,
            'user_id' => $user->id,
            'source_id' => $source->id,
            'source_type' => $source::class,
            'used_at' => null,
        ]);
    }

    public function getTypesWithUser(User $user): Collection
    {
        $types = ScratchCardType::with(['media', 'reason'])
        ->get();

        return $types->map(function (ScratchCardType $type) use ($user) {
            $unusedResultsCount = ScratchCardResult::query()
                ->whereScratchCardTypeId($type->id)
                ->whereUserId($user->id)
                ->whereNull('used_at')
                ->count();

            $canGetScratchCard = $unusedResultsCount >= $type->quantity_to_win;

            $type->current_quantity = $unusedResultsCount;
            $type->can_get_scratch = $canGetScratchCard;
            $type->user = $user;

            return $type;
        });
    }

    public function scratchCardEntryUser(
        User                   $userModel,
        ScratchCardType        $scratchCardTypeModel,
    ): ?ScratchCardEntry
    {
        return DB::transaction(function () use ($userModel, $scratchCardTypeModel) {

            ScratchCardResult::markScratchCardResultsForTargetUser($userModel, $scratchCardTypeModel);

            $prize = $scratchCardTypeModel->getPrizeGenerator()->drawPrize();

            if ($prize->scratchCardPrizeDraw?->type_of_prize === TypeOfPrizeEnum::ELECTRONIC
             && $userModel->pricelessMomentsCard?->card_number === null) {
                $prize->update(['status' => ScratchCardPrizeOrderStatus::class::REJECTED]);
            }

            $prize->assignWinner($userModel);
            return $prize;
        });

    }
}
