<?php

namespace Albedo\Questionnaire\Http\Action;

use Albedo\Questionnaire\Contracts\QuestionnaireUserProvider;
use Albedo\Questionnaire\Http\Requests\SetAnswerRequest;
use Albedo\Questionnaire\Questionnaire;
use Albedo\Questionnaire\QuestionnaireAnswer;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Response;

/**
 * Zapisanie odpowiedzi użytkownika w ankiecie.
 *
 * @group Ankiety
 * Endpoint zapisujący odpowiedzi użytkownika na ankietę.
 *
 * @bodyParam questionnaire_key string required Unikalny klucz ankiety. Example: test-xyz
 * @bodyParam payload object required Odpowiedzi użytkownika (dowolny JSON). Example: {"py1": "TAK"}
 * @response 200 {"answer_id": 1, "payload": {"py1": "TAK"}}
 * @response 403 {"message": "Questionnaire is not active."}
 * @response 404 {"message": "Questionnaire not found."}
 */
class QuestionnaireSetAnswerAction
{
    public function __construct(private QuestionnaireUserProvider $questionnaireUserProvider)
    {
    }

    /**
     * Tworzy lub aktualizuje odpowiedzi użytkownika w ankiecie.
     *
     * @param SetAnswerRequest $request
     * @response 200 {"answer_id": 1, "payload": {"py1": "TAK"}}
     * @response 403 {"message": "Questionnaire is not active."}
     * @response 404 {"message": "Questionnaire not found."}
     */
    public function __invoke(SetAnswerRequest $request): JsonResponse
    {
        $user = $this->questionnaireUserProvider->getCurrentUser();

        if (!$user) {
            return Response::error(message: 'Unauthenticated.', status: 401);
        }

        $validated = $request->validated();

        $questionnaire = Questionnaire::query()
            ->where('unique_key', $validated['questionnaire_key'])
            ->first();

        if (!$questionnaire) {
            return Response::error(message: 'Questionnaire not found.', status: 404);
        }

        if (
            ($questionnaire->start && $questionnaire->start->isFuture()) ||
            ($questionnaire->end && $questionnaire->end->isPast())
        ) {
            return Response::error(message: 'Questionnaire is not active.', status: 403);
        }

        $questionnaireAnswer = QuestionnaireAnswer::query()->updateOrCreate(
            [
                'user_id' => $user->id,
                'questionnaire_id' => $questionnaire->id,
            ],
            [
                'payload' => $validated['payload']
            ]
        );

        if ($questionnaire->payload['is_finished'] ?? false) {
            event(new \Albedo\Questionnaire\Events\UserCompletedQuestionnaireEvent(
                $user,
                $questionnaire,
                $questionnaireAnswer
            ));
        }

        return Response::success(message: 'Answer saved successfully.', data: [
            'answer_id' => $questionnaireAnswer->id,
            'payload' => $questionnaireAnswer->payload,
        ]);
    }
}
