<?php

namespace Albedo\Questionnaire\Filament\Resources;

use Albedo\Questionnaire\Questionnaire;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Tables;
use Albedo\Questionnaire\Filament\Resources\QuestionnaireResource\Pages;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class QuestionnaireResource extends Resource
{
    protected static ?string $model = Questionnaire::class;

    protected static ?string $navigationIcon = 'heroicon-o-question-mark-circle';

    public static function getNavigationBadge(): ?string
    {
        $activeCount = static::getModel()::whereDate('start', '<=', Carbon::now())
            ->whereDate('end', '>=', Carbon::now())
            ->count();

        return $activeCount !== 0 ? 'active' : null;
    }

    public static function getNavigationBadgeColor(): ?string
    {
        return static::getNavigationBadge() > 0 ? 'success' : null;
    }

    public static function form(Forms\Form $form): Forms\Form
    {
        return $form
            ->schema([
                TextInput::make('unique_key')
                    ->unique(ignoreRecord: true)
                    ->default(function ($get) {
                        $name = $get('name');
                        if (!empty($name)) {
                            return Str::slug($name);
                        }
                        return Str::of(Str::random(6))->slug()->upper();
                    })
                    ->afterStateUpdated(function ($set, $get, $state) {
                        if (empty($state) && !empty($get('name'))) {
                            $set('unique_key', Str::slug($get('name')));
                        }
                    }),
                TextInput::make('name')
                    ->required(),

                Forms\Components\Group::make()
                    ->schema([
                        DatePicker::make('start')
                            ->label('Start Date')
                            ->required()
                            ->default(now())
                            ->beforeOrEqual('end'),
                        DatePicker::make('end')
                            ->label('End Date')
                            ->required()
                            ->default(now()->addMonth()->endOfMonth())
                            ->afterOrEqual('start'),
                    ])
                    ->columns(2)
            ])
            ->columns(1);
    }

    public static function table(Tables\Table $table): Tables\Table
    {
        return $table
            ->columns([
                TextColumn::make('name'),
                TextColumn::make('start')
                    ->since()
                    ->dateTimeTooltip(),
                TextColumn::make('end')
                    ->since()
                    ->dateTimeTooltip(),
                TextColumn::make('status')
                    ->label('Status')
                    ->badge()
                    ->getStateUsing(function ($record): string {
                        $now = Carbon::now();
                        if ($record->start <= $now && $record->end >= $now) {
                            return 'Active';
                        }

                        return 'Inactive';
                    })
                    ->colors([
                        'success' => 'Active',
                        'danger' => 'Inactive',
                    ]),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            \Albedo\Questionnaire\Filament\Resources\QuestionnaireResource\RelationManagers\AnswersRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListQuestionnaires::route('/'),
            'create' => Pages\CreateQuestionnaire::route('/create'),
            'edit' => Pages\EditQuestionnaire::route('/{record}/edit'),
        ];
    }
}
