<?php

namespace Albedo\Questionnaire\Http\Action;

use Albedo\Questionnaire\Contracts\QuestionnaireUserProvider;
use Albedo\Questionnaire\Questionnaire;
use Albedo\Questionnaire\QuestionnaireAnswer;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Response;

/**
 * Pobierz aktualny postęp użytkownika w ankiecie.
 *
 * @group Ankiety
 * Endpoint zwraca aktualny zapis lub brak odpowiedzi użytkownika na daną ankietę.
 *
 * @urlParam key string required Unikalny klucz ankiety. Example: test-xyz
 * @response 200 {"payload": {...}}
 * @response 403 {"message": "Questionnaire is not active"}
 * @response 404 {"message": "Questionnaire not found"}
 * @response 409 {"message": "The answer has already been given."}
 */
class QuestionnaireProgressAction
{
    public function __construct(private QuestionnaireUserProvider $questionnaireUserProvider)
    {
    }

    /**
     * Zwraca zapisane odpowiedzi użytkownika, jeśli istnieją.
     *
     * @param string $key
     * @response 200 {"payload": {...}}
     * @response 403 {"message": "Questionnaire is not active"}
     * @response 404 {"message": "Questionnaire not found"}
     * @response 409 {"message": "The answer has already been given."}
     */
    public function __invoke($key): JsonResponse
    {
        $user = $this->questionnaireUserProvider->getCurrentUser();

        if (!$user) {
            return Response::error(message: 'Unauthenticated.', status: 401);
        }

        $questionnaire = Questionnaire::query()
            ->where('unique_key', $key)
            ->first();

        if (!$questionnaire) {
            return Response::success(message:'Questionnaire not found', status: 404);
        }

        if (
            ($questionnaire->start && $questionnaire->start->isFuture()) ||
            ($questionnaire->end && $questionnaire->end->isPast())
        ) {
            return Response::success(message: 'Questionnaire is not active', status: 403);
        }

        $answer = QuestionnaireAnswer::query()
            ->where('user_id', $user->id)
            ->where('questionnaire_id', $questionnaire->id)
            ->first();

        if (($questionnaire->payload['is_finished'] ?? false) === true){
            return Response::success(message: 'The answer has already been given.', status: 409);
        }

        if ($answer) {
            return Response::success(data: [
                'payload' => $answer->payload,
            ]);
        }

        return Response::success(data: [
            'payload' => $questionnaire->payload,
        ]);
    }
}
