<?php

namespace Albedo\Notification\Services;

use Albedo\Notification\Contracts\GroupNotificationInterface;
use Albedo\Notification\Models\NotificationPreferenceDisable;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use ReflectionClass;

class NotificationPreferencesService
{
    public function config(): array
    {
        return [
            'channels' => config('notification.preferences.channels'),
            'groups' => config('notification.preferences.groups'),
            'notifications' => config('notification.preferences.notifications')
        ];
    }

    public function build(Authenticatable $user): array
    {
        return collect($this->config())
            ->map(fn($config) => $this->mapConfig($config, $user))
            ->toArray();
    }

    private function mapConfig(array $config, Authenticatable $user): array
    {
        return collect($config)
            ->reject(fn($data) => ($data['hidden'] ?? false) === true)
            ->map(fn($data, $class) => $this->mapItem($data, $class, $user))
            ->values()
            ->toArray();
    }

    private function mapItem(array $data, string $class, Authenticatable $user): array
    {
        $map = [
            'type' => $data['type'],
            'label' => $data['label'],
            'value' => class_basename($class),
            'description' => $data['description'] ?? null,
            'enabled' => $this->resolveEnabled($class, $user),
            'editable' => $data['editable'] ?? true,
        ];

        if ($this->isGroupNotification($class)) {
            $map['notification'] = $this->mapConfig(
                $data['notifications'] ?? [],
                $user
            );
        }

        return $map;
    }

    private function isGroupNotification(string $class): bool
    {
        return (new ReflectionClass($class))
            ->implementsInterface(GroupNotificationInterface::class);
    }

    private function resolveEnabled(string $class, Authenticatable $user): bool
    {
        return true;
    }

    public function editableValues(): array
    {
        return collect([
            config('notification.preferences.channels'),
            config('notification.preferences.notifications'),
            config('notification.preferences.groups'),
        ])
            ->flatMap(fn($section) => $this->extractEditableValues($section))
            ->unique()
            ->values()
            ->toArray();
    }

    protected function extractEditableValues(array $items): Collection
    {
        return collect($items)->flatMap(
            function (array $data, string $class) {
                $values = collect();

                if (($data['editable'] ?? true) === true) {
                    $values->push(class_basename($class));
                }

                if (!empty($data['notifications']) && is_array($data['notifications'])) {
                    $values = $values->merge(
                        $this->extractEditableValues($data['notifications'])
                    );
                }

                return $values;
            }
        );
    }

    public function updatePreferences(Authenticatable $user, array $preferences): void
    {
        DB::transaction(function () use ($user, $preferences) {

            $disabled = [];
            $enabled = [];

            foreach ($preferences as $class => $value) {
                if ($value === false) {
                    $disabled[] = $class;
                } else {
                    $enabled[] = $class;
                }
            }

            if (!empty($enabled)) {
                NotificationPreferenceDisable::query()
                    ->where('user_id', $user->id)
                    ->whereNotIn('class', $disabled)
                    ->delete();
            }

            if (!empty($disabled)) {

                NotificationPreferenceDisable::query()->upsert(
                    values: collect($disabled)->map(fn($class) => [
                        'user_id' => $user->id,
                        'class' => $class,
                        'enabled' => false,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ])->all(),
                    uniqueBy: ['user_id', 'class'],
                    update: ['enabled', 'updated_at']
                );
            }
        });
    }
}
