<?php

namespace Albedo\LogAlert\Services;

use Albedo\LogAlert\Jobs\LogAlertJob;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class LogAlert
{
    public static function criticalFrontend(
        string  $exception,
        ?string $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Frontend',
            priority: 'critical',
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    public static function criticalBackend(
        Exception $exception,
        ?string   $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Backend',
            priority: 'critical',
            message: $exception->getMessage(),
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    public static function midFrontend(
        string  $exception,
        ?string $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Frontend',
            priority: 'mid',
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    public static function midBackend(
        Exception $exception,
        ?string   $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Backend',
            priority: 'mid',
            message: $exception->getMessage(),
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    public static function lowFrontend(
        string  $exception,
        ?string $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Frontend',
            priority: 'low',
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    public static function lowBackend(
        Exception $exception,
        ?string   $bodyRequest = null
    ): void
    {
        LogAlert::set(
            partOfApplication: 'Backend',
            priority: 'low',
            message: $exception->getMessage(),
            exception: $exception,
            bodyRequest: $bodyRequest
        );
    }

    private static function set(
        string  $partOfApplication,
        string  $priority,
        ?string $message = null,
        string  $exception,
        ?string $bodyRequest = null
    ): void
    {
        $logAlertUrl = config('logalert.url');
        $uniqueKey = config('logalert.unique_key');
        if (
            null !== $uniqueKey
            && null !== $logAlertUrl
        ) {
            \Log::error(['LOGALERT - PATH' => LogAlert::logCallerInfo()]);
            \Log::error(['LOGALERT - EXCEPTION' => $exception]);
            $backtrace = LogAlert::logCallerInfo();

            try {
                DB::connection()->getPdo();
                dispatch(new LogAlertJob(
                    uniqueKey: $uniqueKey,
                    logAlertUrl: $logAlertUrl,
                    partOfApplication: $partOfApplication,
                    priority: $priority,
                    message: $message,
                    exception: $exception,
                    backtrace: $backtrace,
                    bodyRequest: $bodyRequest
                ));

            } catch (\Throwable $e) {
                // send signal about doesn't work DB before sent main log
                LogAlert::sendToDevinci(
                    uniqueKey: $uniqueKey,
                    logAlertUrl: $logAlertUrl,
                    partOfApplication: 'Backend',
                    priority: 'Critical',
                    message: $e->getMessage(),
                    exception: $e,
                    backtrace: $backtrace
                );

                LogAlert::sendToDevinci(
                    uniqueKey: $uniqueKey,
                    logAlertUrl: $logAlertUrl,
                    partOfApplication: $partOfApplication,
                    priority: $priority,
                    message: $message,
                    exception: $exception,
                    backtrace: $backtrace,
                    bodyRequest: $bodyRequest
                );
            }

        }
    }

    public static function sendToDevinci(
        string  $uniqueKey,
        string  $logAlertUrl,
        string  $partOfApplication,
        string  $priority,
        string  $message,
        string  $exception,
        string  $backtrace,
        ?string $bodyRequest = null
    ): void
    {
        try {
            $result = Http::withHeaders([
                'X-Unique-Key' => $uniqueKey,
                'X-Requested-With' => 'XMLHttpRequest'
            ])->post($logAlertUrl, [
                'part_of_application' => $partOfApplication,
                'priority' => $priority,
                'message' => $message,
                'exception' => $exception,
                'backtrace' => $backtrace,
                'body_request' => $bodyRequest
            ]);

            if (in_array($result['code'] ?? null, [400, 404])) {
                Log::error(['ERROR FROM DEVINCI RESPONSE' => $result]);
            }
        } catch (\Exception $exception) {
            Log::info('Something wrong with set log to logAlert');
            Log::error($exception->getMessage());
        }
    }

    private static function logCallerInfo(): string
    {
        $trace = debug_backtrace();
        $caller = $trace[3];

        $class = $caller['class'] ?? null;
        $function = $caller['function'] ?? 'nieznana funkcja';
        $line = $caller['line'] ?? 'nieznana linia';

        return sprintf(
            "Wywołano w: %s",
            ($class ? "$class::$function:$line" : "$function:$line"),
        );
    }
}
