<?php

namespace Albedo\Gamification\Services;

use Albedo\Gamification\Data\ActivityCentreData;
use Albedo\Gamification\Data\AvatarData;
use Albedo\Gamification\Models\ActivityCentre;
use Albedo\Gamification\Models\ActivityTarget;
use Albedo\Gamification\Models\ActivityTargetUserStatus;
use Albedo\Gamification\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Spatie\Permission\Contracts\Role;

class ActivityCentreService
{
    public static function updateOrCreateActivity(ActivityCentreData $data): ActivityCentre
    {
        return ActivityCentre::updateOrCreate([
            'activityable_type' => $data->activityableReferenceData?->type,
            'activityable_id' => $data->activityableReferenceData?->id,
            'type' => $data->type,
        ], [
            'title' => $data->title,
            'body' => $data->body,
            'data' => $data->data,
        ]);
    }

    public static function toEveryone(ActivityCentre $activity): void
    {
        $activity->activityTargets()->updateOrCreate([
            'activity_centre_id' => $activity->getKey(),
            'targetable_type' => 'everyone',
            'targetable_id' => null,
        ]);
    }

    public static function createTarget(ActivityCentre $activity, Model $model): void
    {
        $activity->activityTargets()->updateOrCreate([
            'activity_centre_id' => $activity->getKey(),
            'targetable_type' => $model->getMorphClass(),
            'targetable_id' => $model->getKey(),
        ]);
    }

    public function getUserActivities(User $user, $type, $limit)
    {
        $roles = $user->roles;

        $targets = collect([$user])
            ->merge($roles);

        $paginator = ActivityTarget::query()
            ->with([
                'activityCentre',
                'activityTargetUserStatuses' => fn($q) => $q->whereUserId($user->id)
            ])
            ->whereHas('activityCentre', fn($q) => $q->when($type, fn($q) => $q->whereType($type)))
            ->where(fn($q) => $q
                ->whereMorphedTo(ActivityTarget::TARGETABLE, $targets)
                ->orWhere('targetable_type', 'everyone')
            )
            ->paginate(perPage: $limit);

        $data = $paginator->getCollection()->map(function (ActivityTarget $activityTarget) {

            $isRead = $activityTarget->activityTargetUserStatuses
                ->whereNotNull('read_at')
                ->isNotEmpty();

            return [
                'id' => $activityTarget->id,
                'avatar' => $activityTarget->activityCentre->data['avatar']
                    ?? new AvatarData('system', url('default-avatar.svg'), 'attention'),
                'type' => $activityTarget->activityCentre->type,
                'title' => $activityTarget->activityCentre->title,
                'message' => $activityTarget->activityCentre->body,
                'is_read' => $isRead,
                'images' => [],
                'created_at_formatted' => $activityTarget->activityCentre->created_at->format('d.m.Y'),
            ];
        });

        return [
            'data' => $data,
            'meta' => [
                'current_page' => $paginator->currentPage(),
                'per_page' => $paginator->perPage(),
                'last_page' => $paginator->lastPage(),
                'total' => $paginator->total(),
                'from' => $paginator->firstItem(),
                'to' => $paginator->lastItem(),
                'has_more_pages' => $paginator->hasMorePages(),
            ],
        ];
    }

    public function readUserActivityTargets($user, array $activityIds = []): bool
    {
        $timestamp = now();

        $collection = collect($activityIds)->map(fn($activityId) => [
            'user_id' => $user->id,
            'activity_target_id' => $activityId,
            'read_at' => $timestamp
        ]);

        if ($collection->isEmpty()) {
            return false;
        }

        ActivityTargetUserStatus::query()->upsert(
            $collection->all(),
            ['user_id', 'activity_target_id'],
            ['read_at']
        );

        return true;
    }

    public function unreadUserActivities($user, $type): Collection
    {
        $roles = $user->roles;

        $targets = collect([$user])
            ->merge($roles);

        return ActivityTarget::query()
            ->whereDoesntHave('activityTargetUserStatuses', fn($q) => $q->whereUserId($user->id))
            ->whereHas('activityCentre', fn($q) => $q->when($type, fn($q) => $q->whereType($type)))
            ->where(fn($q) => $q
                ->whereMorphedTo(ActivityTarget::TARGETABLE, $targets)
                ->orWhere('targetable_type', 'everyone')
            )
            ->get();
    }
}
