<?php

namespace Albedo\Gamification\Models;

use Albedo\Gamification\Database\Factories\NewsFactory;
use Albedo\Gamification\Events\NewsPublishedEvent;
use Albedo\Gamification\Events\NewsUnpublishedEvent;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\Permission\Models\Role;

class News extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;

    protected $fillable = [
        'admin_id',
        'title',
        'short_description',
        'content',
        'start_at',
        'end_at',
        'published',
        'url',
        'action_name',
        'for_dashboard',
    ];

    protected function casts(): array
    {
        return [
            'start_at' => 'datetime',
            'end_at' => 'datetime',
            'for_dashboard' => 'boolean',
            'published' => 'boolean',
        ];
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'news_roles');
    }

    public function publish(): bool
    {
        return $this->update([
           'published' => true,
        ]);
    }

    public function unpublish(): bool
    {
        return $this->update([
            'published' => false,
        ]);
    }

    public function isPublished(): bool
    {
        return $this->isDirty(['published'])
            && $this->attributes['published'] === true;
    }

    public function isUnpublished(): bool
    {
        return $this->isDirty(['published'])
            && $this->attributes['published'] === false;
    }

    public function shouldDispatchPublicationEvents()
    {
        if ($this->isPublished()){
            event(new NewsPublishedEvent($this));
        }
        if ($this->isUnpublished()){
            event(new NewsUnpublishedEvent($this));
        }
    }

    protected static function newFactory(): NewsFactory|\Illuminate\Database\Eloquent\Factories\Factory
    {
        return NewsFactory::new();
    }
}
