<?php

namespace Albedo\Gamification;

use Albedo\Gamification\Events\NewsPublishedEvent;
use Albedo\Gamification\Exceptions\GamificationExceptionHandler;
use Albedo\Gamification\Listeners\CreateActivityFromNewsListener;
use Albedo\Gamification\Models\News;
use Albedo\Gamification\Observers\NewsObserver;
use Albedo\Gamification\Providers\Fakers\AnimalFakerProvider;
use Albedo\Gamification\Providers\Fakers\ImageFakerProvider;
use Albedo\Gamification\Providers\Fakers\XxxFakerProvider;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Response;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class GamificationServiceProvider extends PackageServiceProvider
{
    public static string $name = 'gamification';

    public function configurePackage(Package $package): void
    {
        $package
            ->name(static::$name)
            ->hasMigrations([
                '0001_01_01_000000_create_users_table',
                '2022_12_14_083707_create_settings_table',
                '2024_12_30_125323_create_documents_table',
                '2024_12_30_125406_create_agreements_table',
                '2025_10_16_000001_create_user_agreements_table',
                '2025_07_17_081035_create_file_types_table',
                '2025_07_17_081051_create_program_files_table',
                '2025_10_07_115712_create_media_table',
                '2025_07_02_123455_create_avatars_table',
                '2025_10_18_221947_create_user_avatars_table',
                '2025_10_24_085539_create_point_category_table',
                '2025_01_28_143743_create_points_table',
                '2025_11_25_095045_create_rule_contents_table',
                '2025_11_25_130123_create_faqs_table',
                '2025_11_25_132831_create_faq_items_table',
                '2026_01_07_150735_create_activity_centres_table',
                '2026_01_08_102554_create_activity_targets_table',
                '2026_01_08_135942_create_activity_target_user_statuses_table',
                '2025_10_10_092016_create_news_table',
                '2025_10_10_095218_create_news_roles_table',
                '2025_10_27_113615_create_admins_table',
            ])
            ->runsMigrations()
            ->hasConfigFile('gamification')
            ->hasRoute('gamification');

        $this->loadMigrationsFrom(__DIR__ . ' /../database / settings');
    }

    public function bootingPackage()
    {
        $this->app->singleton(ExceptionHandler::class, GamificationExceptionHandler::class);
    }

    public function packageBooted()
    {
        $this->listeners();
        $this->observers();
        $this->fakerBooted();
        $this->responseBooted();
    }

    public function fakerBooted(): self
    {
        $locale = config('app . faker_locale', 'en_US');
        $abstract = \Faker\Generator::class . ':' . $locale;

        $faker = \Faker\Factory::create($locale);

        $providerClass = config('gamification . default_content_provider');

        if ($providerClass && class_exists($providerClass)) {
            $faker->addProvider(new $providerClass($faker));
            $this->app->instance('faker', $faker);
        }

        $faker->addProvider(new ImageFakerProvider($faker));

        App::singleton($abstract, fn() => $faker);

        return $this;
    }

    public function responseBooted(): self
    {
        if (!Response::hasMacro('success')) {
            Response::macro('success', function (mixed $data = [], string $message = '', ?array $errors = null, mixed $meta = [], int $status = 200, array $headers = []) {
                $structure = [
                    'status' => __('OK'),
                    'data' => $data,
                    'meta' => $meta,
                    'message' => $message,
                    'errors' => $errors,
                ];

                return Response::json($structure, $status, $headers);
            });
        }

        if (!Response::hasMacro('error')) {
            Response::macro('error', function (mixed $data = [], string $message = '', mixed $meta = [], ?array $errors = null, int $status = 500, array $headers = []) {
                $structure = [
                    'status' => __('Error'),
                    'data' => $data,
                    'meta' => $meta,
                    'message' => $message,
                    'errors' => $errors,
                ];

                return Response::json($structure, $status, $headers);
            });
        }

        if (!Response::hasMacro('generalError')) {
            Response::macro('generalError', function (mixed $data = [], string $message = '', ?array $errors = null, mixed $meta = [], int $status = 500, array $headers = []) {
                $structure = [
                    'status' => __('Error'),
                    'data' => $data,
                    'meta' => $meta,
                    'message' => $message ?? __('An error occurred'),
                    'errors' => $errors,
                ];

                return Response::json($structure, $status, $headers);
            });
        }

        return $this;
    }

    private function observers(): void
    {
        News::observe(NewsObserver::class);
    }

    /**
     * @return void
     */
    public function listeners(): void
    {
        Event::listen(NewsPublishedEvent::class, CreateActivityFromNewsListener::class);
    }
}
