<?php

namespace Albedo\Duel\Services;

use Albedo\Duel\Models\DuelRound;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Support\Facades\Log;
use SebastianBergmann\Comparator\RuntimeException;

class DuelRoundService
{
    /**
     * @throws \Exception
     */
    public function createDuelRounds(): void
    {
        $period = $this->getDuelRoundPeriod();

        foreach ($period as $date) {
            try {
                $this->createDuelRoundForDate($date);
            } catch (\Exception $e) {
                Log::error('Failed to create duel round', [
                    'date' => $date,
                    'error' => $e->getMessage()
                ]);

                throw $e;
            }
        }
    }

    public function getDuelRoundPeriod(): CarbonPeriod
    {
        $startDate = $this->getStartDate();
        $endDate = $this->getEndDate();

        return CarbonPeriod::create($startDate, '7 days', $endDate);
    }

    public function getStartDate(): Carbon
    {
        $start = config('duel.start_duel_round');

        if ($start) {
            return Carbon::parse($start)->startOfWeek(); // Zawsze zaczynamy od poniedziałku
        }

        return Carbon::now()->startOfWeek()->startOfDay();
    }

    public function getEndDate(): Carbon
    {
        $end = config('duel.end_duel_round');

        if ($end) {
            return Carbon::parse($end)->endOfDay();
        }

        // Domyślnie kończymy w piątek bieżącego tygodnia
        return Carbon::now()->endOfWeek()->subDays(2)->endOfDay(); // Piątek bieżącego tygodnia
    }

    public function createDuelRoundForDate(Carbon $date): void
    {
        // Zawsze poniedziałek jako start i piątek jako koniec
        $startDate = $date->copy()->startOfWeek(); // Poniedziałek
        $endDate = $date->copy()->endOfWeek()->subDays(2)->endOfDay(); // Piątek 23:59:59

        if ($this->duelRoundExists($startDate, $endDate)) {
            throw new \Exception('Duel round already exists for period: ' . $startDate->toDateString() . ' - ' . $endDate->toDateString());
        }

        DuelRound::create([
            'name' => $this->generateRoundName(),
            'begin_date' => $startDate,
            'end_date' => $endDate,
        ]);
    }

    public function duelRoundExists(Carbon $startDate, Carbon $endDate): bool
    {
        return DuelRound::query()
            ->whereDate('begin_date', '>=', $startDate)
            ->whereDate('end_date', '<=', $endDate)
            ->exists();
    }

    public function generateRoundName(): string
    {
        $roundCount = DuelRound::count();
        return sprintf('Round %s', $roundCount + 1);
    }

    public function getDuelRoundCurrent(): ?DuelRound
    {
        $duelRound = DuelRound::getCurrent();

        if (!$duelRound) {
            throw new RuntimeException('No active round found.');
        }

        return $duelRound;
    }
}
