<?php

namespace Albedo\Duel\Http\Controllers;

use Albedo\Duel\Services\NextDuelQuestionService;
use Albedo\Duel\Exceptions\DuelQuestionNotFoundException;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Settings\DuelSettings;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;


class DuelQuestionAction extends Controller
{
    public function __construct(
        protected DuelSettings $duelSettings,
        protected NextDuelQuestionService $getNextDuelQuestion
    )
    {

    }

    /**
     * Pytanie pojedynkowe
     *
     * Endpoint zwraca kolejne pytanie do rozwiązania dla zalogowanego użytkownika w ramach wskazanego pojedynku.
     *
     * @group Pojedynki
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     *
     * @response 200 scenario="Jest dostępne pytanie" {
     *   "status": "OK",
     *   "data": {
     *     "id": 1,
     *     "question": "Jak nazywa się stolica Polski?",
     *     "number": 1,
     *     "number_by_category": 0,
     *     "category": "Geografia",
     *     "options": [
     *       {"value": "A", "label": "Warszawa"},
     *       {"value": "B", "label": "Kraków"},
     *       {"value": "C", "label": "Gdańsk"}
     *     ]
     *   },
     *   "message": "Success"
     * }
     * @response 200 scenario="Brak pytań do rozwiązania" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "No questions in the duel"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function __invoke(Duel $duel): JsonResponse
    {
        $userId = (int) auth()->id();

        try {
            $questionResource = $this->getNextDuelQuestion->handle($duel, $userId);
        } catch(DuelQuestionNotFoundException $ex) {
            return Response::error(message: $ex->getMessage(), status: 404);
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }
        return Response::success(data: $questionResource, message: "Success");
    }
}
