<?php

namespace Albedo\Duel\Models;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultEnum;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Albedo\Duel\Events\DuelUserCreatedEvent;
use Albedo\Gamification\Models\Traits\PointableTrait;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class DuelUser extends Model
{
    use HasFactory;

    protected $dispatchesEvents = [
        'created' => DuelUserCreatedEvent::class
    ];

    protected $fillable = [
        'duel_id',
        'user_id',
        'started_at',
        'ended_at',
        'progress_status',
        'result_status',
        'points',
        'time',
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'ended_at' => 'datetime',
        'progress_status' => DuelProgressStatusEnum::class,
        'result_status' => DuelResultEnum::class,
    ];

    public function duel(): BelongsTo
    {
        return $this->belongsTo(Duel::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(config('duel.user_model'));
    }

    public function userWithoutGlobalScopes(): BelongsTo
    {
        return $this->user()->withoutGlobalScopes();
    }

    public function duelOpponent(): HasOne
    {
        return $this->hasOne(__CLASS__, 'duel_id', 'duel_id')
            ->where('user_id', '!=', $this->user_id);
    }

    public function duelUserAnswers(): HasMany
    {
        return $this->hasMany(DuelUserAnswer::class);
    }

    public function hasParticipated()
    {
        return in_array($this->progress_status, [
            DuelProgressStatusEnum::IN_PROGRESS,
            DuelProgressStatusEnum::FINISHED
        ]);
    }

    public function notParticipated()
    {
        return $this->progress_status === DuelProgressStatusEnum::NOT_PARTICIPATED;
    }

    public function time(): Attribute
    {
        if ($this->duelUserAnswers
            ->reject(fn(DuelUserAnswer $duelUserAnswer) => is_null($duelUserAnswer->answer_expiration_date))->isEmpty()) {
            return Attribute::make();
        }

        return Attribute::make(
            get: fn() => $this->duelUserAnswers
                ->sum(fn(duelUserAnswer $duelUserAnswer) => $duelUserAnswer->answer_time ?? 46 * 1000));
    }

    public function formattedTime(): string
    {
        $seconds = $this->time / 1000;
        $minutes = intdiv($seconds, 60);
        $remainingSeconds = $seconds % 60;

        return sprintf('%02d:%02d', $minutes, $remainingSeconds);
    }

    public function isFinished(): bool
    {
        return $this->progress_status === DuelProgressStatusEnum::FINISHED;
    }

    public function countCorrectAnswers()
    {
        return $this->duelUserAnswers
            ->filter(fn(DuelUserAnswer $duelUserAnswer) => $duelUserAnswer->isCorrect())
            ->count();
    }

    public function getSumTimeFormatted(): string
    {
        return $this->getSumTimeSeconds();
    }

    public function getSumTimeSeconds(): int
    {
        return $this->duelUserAnswers
            ->each(fn(duelUserAnswer $duelUserAnswer) => $this->updateAnswerTimeIsNull($duelUserAnswer))
            ->sum(fn(duelUserAnswer $duelUserAnswer) => $this->getAnswerTime($duelUserAnswer));
    }

    /**
     * @param DuelUserAnswer $duelUserAnswer
     */
    public function updateAnswerTimeIsNull(DuelUserAnswer $duelUserAnswer): void
    {
            $duelUserAnswer->answer_time ?? $duelUserAnswer->update(['answer_time' => $this->getMaxTime()]);
    }

    public function getMaxTime(): int
    {
        return DuelService::getMaxTotalAnswerTimeInMs(app(DuelSettings::class));
    }

    public function getAnswerTime(duelUserAnswer $duelUserAnswer): int
    {
        return $duelUserAnswer->answer_time ?? $this->maxTotalAnswerTime;
    }

    public function getDateForPeriod()
    {
        return $this->duel?->duel_round?->begin_date ?? $this->created_at;
    }

    protected static function newFactory()
    {
        return \Albedo\Duel\Database\Factories\DuelUserFactory::new();
    }
}
