<?php

namespace Albedo\Duel\Application;

use Albedo\Duel\Domain\Exceptions\AnswerAlreadyGivenException;
use Albedo\Duel\Domain\Exceptions\AnswerTooLateException;
use Albedo\Duel\Domain\Exceptions\DuelQuestionNotFoundException;
use Albedo\Duel\Events\DuelCorrectAnswerGivenEvent;
use Albedo\Duel\Events\DuelPossibleEndEvent;
use Albedo\Duel\Events\DuelQuestionAnsweredEvent;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Contracts\Events\Dispatcher;

class AnswerDuelQuestion
{
    public function __construct(
        protected DuelSettings $duelSettings,
        protected Dispatcher $events
    ) {}

    public function execute(Duel $duel, int $userId, int $questionId, int $answerId): void
    {
            $now = now();

            $duelUserAnswer = DuelUserAnswer::query()
                ->lockForUpdate()
                ->whereDuelId($duel->id)
                ->whereRelation('duelUser', 'user_id', $userId)
                ->whereDuelQuestionId($questionId)
                ->whereNotNull('answer_expiration_date')
                ->orderBy('number')
                ->first();

            if (!$duelUserAnswer) {
                throw new DuelQuestionNotFoundException("I couldn't find a question associated with this duel");
            }

            if ($duelUserAnswer->duel_answer_id) {
                throw new AnswerAlreadyGivenException('The answer to this question has already been given');
            }

            $maxAnswerTime = $this->duelSettings->answer_time_absolute_in_seconds * 1000;

            $duelUserAnswer->answer_time = $maxAnswerTime - $now->diffInMilliseconds($duelUserAnswer->answer_expiration_date);
            $duelUserAnswer->duel_answer_id = $answerId;

            if ($duelUserAnswer->answer_expiration_date <= $now) {
                $duelUserAnswer->answer_time = $maxAnswerTime;
                $duelUserAnswer->timeout = true;
                $duelUserAnswer->save();
                $this->events->dispatch(new DuelPossibleEndEvent($duel));
                throw new AnswerTooLateException('The answer to this question was given too late');
            }

            if ($duelUserAnswer->answer_time > $maxAnswerTime) {
                $duelUserAnswer->answer_time = $maxAnswerTime;
            }

            $duelUserAnswer->save();

            $duelUserAnswer->refresh();

            $duelUser = $duelUserAnswer->duelUser;
            if ($duelUserAnswer->duelAnswer?->is_valid && $duelUserAnswer->duelAnswer?->duel_question_id == $duelUserAnswer->duel_question_id) {
                $duelUser->points += $this->duelSettings->points_for_correct_answer;
                $duelUserAnswer->is_correct = true;
                $duelUserAnswer->save();
                $this->events->dispatch(new DuelCorrectAnswerGivenEvent($duelUserAnswer));
            } else {
                $duelUserAnswer->is_correct = false;
                $duelUserAnswer->save();
            }

            $duelUser->ended_at = $now;
            $duelUser->save();

            $this->events->dispatch(new DuelQuestionAnsweredEvent($duelUserAnswer));
            $this->events->dispatch(new DuelPossibleEndEvent($duel));
    }
}


