<?php

namespace Albedo\Duel\Commands;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultStatusEnum;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelQuestion;
use Albedo\Duel\Models\DuelRound;
use Albedo\Duel\Models\DuelUser;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class MakeUserDuelsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'userDuel:make';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';
    /**
     * @var DuelSettings|\Illuminate\Foundation\Application|mixed|object
     */
    protected DuelSettings $duelSettings;

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $this->line('Start');


        $duelRound = DuelRound::getCurrent();

        if (is_null($duelRound)) {
            $this->error('No active round found.');
            return;
        }

        $this->duelSettings = app(DuelSettings::class);
        $duelService = app(DuelService::class);

        $participants = $duelService->getParticipantsWithActiveDuel();
        $participants = $this->filterParticipantsWithoutActiveDuel($duelRound, $participants);
        $participants = $this->shuffleParticipants($participants);
        $botUser = $this->getParticipantBot();

        while ($participants->count() > 0) {
            $duel = $this->createDuel($duelRound);
            $duelQuestions = $this->getDuelQuestions();

            $allKeys = $participants->keys();
            $key = $allKeys[0];

            $duelUserFirst = $this->createDuelUser($duel, $participants[$key]);

            $opponentKey = null;
            $shouldHaveBot = true;

            foreach ($allKeys as $anotherKey) {
                if ($anotherKey == $key) {
                    continue;
                }

                if ($participants[$anotherKey]->vehicle?->id != $participants[$key]->vehicle?->id) {
                    $opponentKey = $anotherKey;
                    $shouldHaveBot = false;
                    break;
                }
            }

            if ($shouldHaveBot) {
                $duelUserOpponent = $this->createDuelUser($duel, $botUser);
            } else {
                $duelUserOpponent = $this->createDuelUser($duel, $participants[$opponentKey]);
            }


            $numberQuestionFirst = 1;
            $numberQuestionOpponent = 1;
            foreach ($duelQuestions as $questions) {
                $numberByCategoryFirst = 1;
                $numberByCategoryOpponent = 1;
                foreach ($questions as $question) {
                    $this->createDuelUserAnswer(
                        $duel,
                        $duelUserFirst,
                        $question,
                        $numberQuestionFirst,
                        $numberByCategoryFirst
                    );
                    $this->createDuelUserAnswer(
                        $duel,
                        $duelUserOpponent,
                        $question,
                        $numberQuestionOpponent,
                        $numberByCategoryOpponent
                    );
                }
            }

            $participants->pull($key);

            if (!$shouldHaveBot) {
                $participants->pull($opponentKey);
            }

        }

        $this->line('End');
    }

    /**
     * @return Model
     */
    private function getParticipantBot(): Model
    {
        return config('duel.user_model')::query()
            ->whereIsBot(true)
            ->firstOr(callback: function () {
                return config('duel.user_model')::firstOrCreate([
                    'is_bot' => true,
                    'email' => 'j.albedo@albedomarketing.pl',
                ], [
                    'first_name' => 'Jan',
                    'last_name' => 'Albedo',
                    'email_verified_at' => now(),
                    'password' => Str::password(),
                ]);
            });
    }


    /**
     * @param DuelRound $duelRound
     * @param Collection $participants
     * @return Collection
     */
    public function filterParticipantsWithoutActiveDuel(DuelRound $duelRound, Collection $participants): Collection
    {
        $duelUsersHasRound = DuelUser::query()
            ->whereHas('duel', fn($q) => $q->where('duel_round_id', $duelRound->id))
            ->get();

        return $participants->reject(function (User $user) use ($duelUsersHasRound) {
            $result = $duelUsersHasRound->contains('user_id', $user->id);
            if ($result) {
                $this->info($user->getKey() . ' already has an active duel.');
            }
            return $result;
        });
    }

    /**
     * @param Collection $participants
     * @return Collection
     */
    public function shuffleParticipants(Collection $participants): Collection
    {
        return $participants->shuffle();
    }

    /**
     * @param DuelRound $duelRound
     * @return Duel
     */
    public function createDuel(DuelRound $duelRound): Duel
    {
        $this->line("Create duel");
        return Duel::create([
            'duel_round_id' => $duelRound->id,
            'progress_status' => DuelProgressStatusEnum::NOT_PARTICIPATED,
        ]);
    }

    /**
     * @param DuelRound $duelRound
     * @param int $totalQuestionsByCategory
     * @return Collection
     */
    public function getDuelQuestions(): Collection
    {
        return DuelQuestion::query()
            ->whereIsActive(true)
            ->get()
            ->groupBy('duel_question_category_id')
            ->map(function ($collection) {
                $quantityInDuel = $collection->this->duelSettings->total_questions_by_category;
                return $collection
                    ->shuffle()
                    ->slice(0, $quantityInDuel);
            });
    }

    /**
     * @param Duel $duel
     * @param $user
     * @return DuelUser
     */
    public function createDuelUser(Duel $duel, $user): DuelUser
    {
        $this->line("Adding player: " . $user->getKey());

        return DuelUser::create([
            'duel_id' => $duel->id,
            'user_id' => $user->id,
            'progress_status' => DuelProgressStatusEnum::NOT_PARTICIPATED,
            'result_status' => DuelResultStatusEnum::UNKNOWN,
            'points' => 0,
        ]);
    }

    /**
     * @param Duel $duel
     * @param DuelUser $duelUser
     * @param mixed $question
     * @param int $numberQuestion
     * @param int $numberByCategory
     * @return void
     */
    public function createDuelUserAnswer(Duel $duel, DuelUser $duelUser, mixed $question, int &$numberQuestion, int &$numberByCategory): DuelUserAnswer
    {
        return DuelUserAnswer::create([
            'duel_id' => $duel->id,
            'duel_user_id' => $duelUser->id,
            'duel_question_id' => $question->id,
            'number' => $numberQuestion++,
            'number_by_category' => $numberByCategory++,
        ]);
    }
}
