<?php

namespace Albedo\Duel\Http\Controllers;

use Albedo\Duel\Events\DuelCorrectAnswerGivenEvent;
use Albedo\Duel\Events\DuelPossibleEndEvent;
use Albedo\Duel\Events\DuelQuestionAnsweredEvent;
use Albedo\Duel\Http\Requests\DuelAnswerRequest;
use Albedo\Duel\Http\Resources\DuelResource;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Exception;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * @group Duel
 */
class DuelController extends Controller
{

    public function __construct(
        protected Authenticatable $user,
        protected DuelSettings    $duelSettings
    )
    {
    }

    /**
     * @group Pojedynki
     *
     * Zapisz odpowiedź na pytanie w pojedynku
     *
     * Endpoint zapisuje odpowiedź użytkownika na konkretne pytanie w ramach wskazanego pojedynku.
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     * @bodyParam question_id int required ID pytania. Example: 5
     * @bodyParam answer_id int required ID wybranej odpowiedzi. Example: 12
     *
     * @response 200 scenario="Odpowiedź zapisana poprawnie" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "Answer has been saved"
     * }
     * @response 400 scenario="Brak pytania lub już udzielono odpowiedzi" {
     *   "status": "ERROR",
     *   "message": "I couldn't find a question associated with this duel"
     * }
     * @response 400 scenario="Odpowiedź już udzielona" {
     *   "status": "ERROR",
     *   "message": "The answer to this question has already been given"
     * }
     * @response 400 scenario="Za późno na odpowiedź" {
     *   "status": "ERROR",
     *   "message": "The answer to this question was given too late"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function storeAnswer(Duel $duel, DuelAnswerRequest $request): JsonResponse
    {
        $now = now();
        $questionId = $request->input('question_id');
        $answerId = $request->input('answer_id');

        $duelUserAnswer = DuelUserAnswer::query()
            ->lockForUpdate()
            ->whereDuelId($duel->id)
            ->whereRelation('duelUser', 'user_id', $this->user->getKey())
            ->whereDuelQuestionId($questionId)
            ->whereNotNull('answer_expiration_date')
            ->orderBy('number')
            ->first();

        if (!$duelUserAnswer) {
            return Response::error(message: 'I couldn\'t find a question associated with this duel');
        }

        if ($duelUserAnswer->duel_answer_id) {
            return Response::error(message: 'The answer to this question has already been given');
        }

        $maxAnswerTime = $this->duelSettings->answer_time_absolute_in_seconds * 1000;
        $maxTotalTime = DuelService::getMaxTotalAnswerTimeInMs($this->duelSettings);

        if ($duelUserAnswer->answer_expiration_date <= $now) {
            $duelUserAnswer->answer_time = $maxTotalTime - $now->diffInMilliseconds($duelUserAnswer->answer_expiration_date);
            $duelUserAnswer->save();
            return Response::error(message: 'The answer to this question was given too late');
        }

        $duelUserAnswer->duel_answer_id = $answerId;
        $duelUserAnswer->answer_time = $maxTotalTime - $now->diffInMilliseconds($duelUserAnswer->answer_expiration_date);

        if ($duelUserAnswer->answer_time > $maxAnswerTime) {
            $duelUserAnswer->answer_time = $maxAnswerTime;
        }

        $duelUserAnswer->save();

        $duelUserAnswer = $duelUserAnswer->fresh();

        $duelUser = $duelUserAnswer->duelUser;
        if ($duelUserAnswer->duelAnswer?->is_valid && $duelUserAnswer->duelAnswer?->duel_question_id == $duelUserAnswer->duel_question_id) {
            $duelUser->points += $this->duelSettings->points_for_correct_answer;
            $duelUserAnswer->is_correct = true;
            $duelUserAnswer->save();
            event(new DuelCorrectAnswerGivenEvent($duelUserAnswer));
        }

        $duelUser->ended_at = $now;
        $duelUser->save();

        event(new DuelQuestionAnsweredEvent($duelUserAnswer));
        event(new DuelPossibleEndEvent($duel));

        return Response::success(message: 'Answer has been saved');
    }

    /**
     * @group Pojedynki
     *
     * Pobierz podsumowanie pojedynku
     *
     * Endpoint zwraca podsumowanie wskazanego pojedynku dla zalogowanego użytkownika.
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     *
     * @response 200 scenario="Podsumowanie pojedynku" {
     *   "status": "OK",
     *   "data": {
     *     "id": 1,
     *     "name": "Runda X",
     *     "begin_date": "27.10",
     *     "end_date": "31.10.2025",
     *     "status": "finished",
     *     "me": {
     *       "first_name": "Santos",
     *       "last_name": "U.",
     *       "name": "Deleted U.",
     *       "time": "00:00",
     *       "points": null,
     *       "answers": [ ... ]
     *     },
     *     "opponent": {
     *       "first_name": "Lyda",
     *       "last_name": "U.",
     *       "name": "Deleted U.",
     *       "time": "00:00",
     *       "points": null,
     *       "answers": [ ... ]
     *     }
     *   },
     *   "message": "Success"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function summary(Duel $duel)
    {
        try {
            return Response::success(data: DuelResource::make($duel), message: "Success");
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }
    }
}
