<?php

namespace Albedo\Duel\Commands;

use Albedo\Ranking\Rankings\IndividualStageDuelRanking;
use Albedo\Ranking\Rankings\TeamStageCountryRanking;
use App\Events\DuelAndChallengeEndedEvent;
use App\Events\DuelCloseEvent;
use App\Models\Duel;
use App\Models\DuelRound;
use App\Models\Stage;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CloseUserDuelsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'userDuel:close';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->line('Start');
        Log::info('Start of close user duels');

        $processedDuelRounds = [];

        Duel::whereStatus(Duel::STATUS_IN_PROGRESS)
            ->get()
            ->map(function (Duel $duel) use (&$processedDuelRounds) {
                $this->line('ID - ' . $duel->id);
                Log::info('Closing Duel ID - ' . $duel->id);
                try {

                    if (!in_array($duel->duel_round_id, $processedDuelRounds)) {
                        $processedDuelRounds[] = $duel->duel_round_id;
                    }

                    event(new DuelCloseEvent($duel));
                    event(new DuelAndChallengeEndedEvent($duel));
                } catch (\Throwable $ex) {
                    Log::error('Error while closing duel ID - ' . $duel->id);
                    report($ex);
                }
                $this->line('Zakończono');
                Log::info('Duel ID - ' . $duel->id . ' closed');
            });

        foreach ($processedDuelRounds as $duelRoundId) {
            try {
                $duelRound = DuelRound::find($duelRoundId);

                if (!$duelRound) {
                    continue;
                }
                Log::info('Generating ranking for Duel Round ID - ' . $duelRoundId);
                (new IndividualStageDuelRanking($duelRound))->generate();

                $stage = Stage::getPeriodByDate($duelRound->begin_date);
                if ($stage) {
                    Log::info('Generating ranking for Stage ID - ' . $stage->id);
                    (new TeamStageCountryRanking($stage))->generate();
                }
            } catch (Exception $ex) {
                Log::error('Error while generating ranking while closing Duel Round ID - ' . $duelRoundId);
                $this->line($ex->getMessage());
                report($ex);
            }

        }

        $this->line('Koniec');
        Log::info('End of close user duels');
    }
}
