<?php

namespace Albedo\Duel\Http\Controllers;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Events\DuelNextQuestionRetrievedEvent;
use Albedo\Duel\Events\DuelPossibleEndEvent;
use Albedo\Duel\Http\Resources\QuestionResource;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Routing\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Response;


class DuelQuestionAction extends Controller
{
    public function __construct(protected Authenticatable $user, protected DuelSettings $duelSettings)
    {

    }

    /**
     * Pytanie pojedynkowe
     *
     * Endpoint zwraca kolejne pytanie do rozwiązania dla zalogowanego użytkownika w ramach wskazanego pojedynku.
     *
     * @group Pojedynki
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     *
     * @response 200 scenario="Jest dostępne pytanie" {
     *   "status": "OK",
     *   "data": {
     *     "id": 1,
     *     "question": "Jak nazywa się stolica Polski?",
     *     "number": 1,
     *     "number_by_category": 0,
     *     "category": "Geografia",
     *     "options": [
     *       {"value": "A", "label": "Warszawa"},
     *       {"value": "B", "label": "Kraków"},
     *       {"value": "C", "label": "Gdańsk"}
     *     ]
     *   },
     *   "message": "Success"
     * }
     * @response 200 scenario="Brak pytań do rozwiązania" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "No questions in the duel"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function __invoke(Duel $duel): JsonResponse
    {
        $duelUserAnswer = DuelUserAnswer::where('duel_id', $duel->id)
            ->whereHas('duelUser', function ($q) {
                $q->where('user_id', $this->user->id);
            })->whereNull('answer_expiration_date')
            ->orderBy('number')
            ->first();

        if (!$duelUserAnswer) {
            event(new DuelPossibleEndEvent($duel));
            return Response::success(message:'No questions in the duel');
        }

        $maxTotalAnswerTime = DuelService::getMaxTotalAnswerTimeInMs($this->duelSettings);

        if ($duelUserAnswer->number === 1) {
            $duelUserAnswer->duelUser()->update([
                'started_at' => now(),
                'progress_status' => DuelProgressStatusEnum::IN_PROGRESS,
            ]);
            $duelUserAnswer->duel()->update([
                'progress_status' => DuelProgressStatusEnum::IN_PROGRESS,
            ]);
        }

        $duelUserAnswer->answer_expiration_date = now()->addMillis($maxTotalAnswerTime);
        $duelUserAnswer->save();
        $duelUserAnswer->duelQuestion->number = $duelUserAnswer->number;
        $duelUserAnswer->duelQuestion->number_by_category = $duelUserAnswer->number_by_category;

        event(new DuelNextQuestionRetrievedEvent($duelUserAnswer));

        return Response::success(data: QuestionResource::make($duelUserAnswer->duelQuestion), message: "Success");
    }
}
