<?php

namespace Albedo\Duel\Services;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultStatusEnum;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelQuestion;
use Albedo\Duel\Models\DuelRound;
use Albedo\Duel\Models\DuelUser;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Support\Collection;

class DuelGeneratorService
{
    public function __construct(
        protected DuelSettings     $settings,
        protected DuelRoundService $duelRoundService,
    )
    {
    }

    public function generate(): void
    {
        $round = $this->duelRoundService->getDuelRoundCurrent();

        $participants = $this->getParticipants();
        $participants = $this->filterParticipantsWithoutActiveDuel($round, $participants);
        $participants = $participants->shuffle();

        $bot = $this->getOrCreateBot();

        while ($participants->count() > 0) {

            $duel = $this->createDuel($round);
            $questions = $this->getDuelQuestions();

            $first = $participants->shift();

            $opponent = $this->findOpponent($first, $participants, $bot);

            if (!$opponent->is_bot) {
                $participants = $participants->reject(fn($u) => $u->id === $opponent->id)->values();
            }

            $firstDuelUser = $this->createDuelUser($duel, $first);
            $secondDuelUser = $this->createDuelUser($duel, $opponent);

            $this->assignQuestions($duel, $firstDuelUser, $secondDuelUser, $questions);
        }
    }

    public function getParticipants(): Collection
    {
        $model = config('duel.user_model');

        return $model::query()
            ->whereIsBot(false)
            ->get();
    }

    private function getOrCreateBot()
    {
        $model = config('duel.user_model');

        return $model::query()
            ->whereIsBot(true)
            ->firstOr(function () use ($model) {
                $model = $model::firstOrCreate([
                    'is_bot' => true,
                    'email' => 'albedo@albedomarketing.pl'
                ], [
                    'first_name' => 'Albert',
                    'last_name' => 'Albedo',
                    'email_verified_at' => now(),
                    'password' => bcrypt(str()->random()),
                ]);

                if (!$model->is_bot) {
                    $model->forceFill(['is_bot' => true])->save();
                }

                return $model;
            });
    }

    private function filterParticipantsWithoutActiveDuel(DuelRound $round, Collection $participants): Collection
    {
        $duelUsersHasRound = DuelUser::query()
            ->whereHas('duel', fn($q) => $q->where('duel_round_id', $round->id))
            ->get();

        return $participants->reject(fn($u) => $duelUsersHasRound->contains('user_id', $u->id));
    }

    private function createDuel(DuelRound $round): Duel
    {
        return Duel::create([
            'duel_round_id' => $round->id,
            'progress_status' => DuelProgressStatusEnum::NOT_PARTICIPATED,
        ]);
    }

    private function getDuelQuestions(): Collection
    {
        return DuelQuestion::query()
            ->whereIsActive(true)
            ->get()
            ->groupBy('duel_question_category_id')
            ->map(fn($group) => $group
                ->shuffle()
                ->take($this->settings->total_questions_by_category));
    }

    private function createDuelUser(Duel $duel, $user): DuelUser
    {
        return DuelUser::create([
            'duel_id' => $duel->id,
            'user_id' => $user->id,
            'progress_status' => DuelProgressStatusEnum::NOT_PARTICIPATED,
            'result_status' => DuelResultStatusEnum::UNKNOWN,
            'points' => 0,
        ]);
    }

    private function assignQuestions(
        Duel       $duel,
        DuelUser   $first,
        DuelUser   $second,
        Collection $questions
    ): void
    {
        $firstCounter = 1;
        $secondCounter = 1;

        foreach ($questions as $categoryQuestions) {
            $firstByCat = 1;
            $secondByCat = 1;

            foreach ($categoryQuestions as $question) {
                DuelUserAnswer::create([
                    'duel_id' => $duel->id,
                    'duel_user_id' => $first->id,
                    'duel_question_id' => $question->id,
                    'number' => $firstCounter++,
                    'number_by_category' => $firstByCat++,
                ]);

                DuelUserAnswer::create([
                    'duel_id' => $duel->id,
                    'duel_user_id' => $second->id,
                    'duel_question_id' => $question->id,
                    'number' => $secondCounter++,
                    'number_by_category' => $secondByCat++,
                ]);
            }
        }
    }

    public function findOpponent($first, Collection $participants, $bot)
    {
        foreach ($participants as $p) {
            if ($p->id !== $first->id) {
                return $p;
            }
        }

        return $bot;
    }
}
