<?php

namespace Albedo\Duel\Models;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultStatusEnum;
use Albedo\Duel\Events\DuelUserCreatedEvent;
use Albedo\Duel\Models\Traits\CalculatesDuelStats;
use Albedo\Duel\Models\Traits\DuelAnswerTime;
use Albedo\Duel\Models\Traits\HasDuelProgressStatus;
use Albedo\Duel\Models\Traits\HasDuelResultStatus;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class DuelUser extends Model
{
    use HasFactory,
        SoftDeletes,
        HasDuelProgressStatus,
        HasDuelResultStatus,
        DuelAnswerTime,
        CalculatesDuelStats;

    protected $dispatchesEvents = [
        'created' => DuelUserCreatedEvent::class
    ];

    protected $fillable = [
        'duel_id',
        'user_id',
        'started_at',
        'ended_at',
        'progress_status',
        'result_status',
        'points',
        'time',
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'ended_at' => 'datetime',
        'progress_status' => DuelProgressStatusEnum::class,
        'result_status' => DuelResultStatusEnum::class,
    ];

    public function duel(): BelongsTo
    {
        return $this->belongsTo(Duel::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(config('duel.user_model'));
    }

    public function userWithoutGlobalScopes(): BelongsTo
    {
        return $this->user()->withoutGlobalScopes();
    }

    public function duelOpponent(): HasOne
    {
        return $this->hasOne(__CLASS__, 'duel_id', 'duel_id')
            ->where('user_id', '!=', $this->user_id);
    }

    public function duelUserAnswers(): HasMany
    {
        return $this->hasMany(DuelUserAnswer::class);
    }


    protected static function newFactory()
    {
        return \Albedo\Duel\Database\Factories\DuelUserFactory::new();
    }
}
