<?php

namespace Albedo\Duel\Services;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Events\DuelNextQuestionRetrievedEvent;
use Albedo\Duel\Events\DuelPossibleEndEvent;
use Albedo\Duel\Http\Resources\QuestionResource;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Contracts\Events\Dispatcher;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Exceptions\DuelQuestionNotFoundException;

class NextDuelQuestionService
{
    public function __construct(
        protected DuelSettings $duelSettings,
        protected Dispatcher $events
    ) {}

    public function handle(Duel $duel, int $userId): ?QuestionResource
    {
        $now = now();

        $duelUserAnswer = DuelUserAnswer::where('duel_id', $duel->id)
            ->whereHas('duelUser', function ($q) use ($userId) {
                $q->where('user_id', $userId);
            })
            ->whereNull('answer_expiration_date')
            ->orderBy('number')
            ->first();

        if (!$duelUserAnswer) {
            $this->events->dispatch(new DuelPossibleEndEvent($duel));
            throw new DuelQuestionNotFoundException();
        }

        $maxTotalAnswerTime = DuelService::getMaxTotalAnswerTimeInMs($this->duelSettings);

        if ($duelUserAnswer->number === 1) {
            $duelUserAnswer->duelUser()->update([
                'started_at' => now(),
                'progress_status' => DuelProgressStatusEnum::IN_PROGRESS,
            ]);
            $duelUserAnswer->duel()->update([
                'progress_status' => DuelProgressStatusEnum::IN_PROGRESS,
            ]);
        }

        $duelUserAnswer->answer_expiration_date = $now->copy()->addMillis($maxTotalAnswerTime);
        $duelUserAnswer->save();
        $duelUserAnswer->duelQuestion->number = $duelUserAnswer->number;
        $duelUserAnswer->duelQuestion->number_by_category = $duelUserAnswer->number_by_category;

        $this->events->dispatch(new DuelNextQuestionRetrievedEvent($duelUserAnswer));

        return QuestionResource::make($duelUserAnswer->duelQuestion);
    }
}
