<?php

namespace Albedo\Duel\Http\Controllers;


use Albedo\Duel\Services\AnswerDuelQuestionService;
use Albedo\Duel\Exceptions\AnswerAlreadyGivenException;
use Albedo\Duel\Exceptions\AnswerTimeoutException;
use Albedo\Duel\Exceptions\DuelQuestionNotFoundException;
use Albedo\Duel\Http\Requests\DuelAnswerRequest;
use Albedo\Duel\Models\Duel;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

class DuelAnswerAction extends Controller
{
    public function __construct(
        protected AnswerDuelQuestionService $answerDuelQuestion
    ) {}

    /**
     * Odpowiedź pojedynkowa
     *
     * Zapisuje odpowiedź użytkownika na konkretne pytanie w ramach wskazanego pojedynku.
     *
     * @group Pojedynki
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     * @bodyParam question_id int required ID pytania. Example: 5
     * @bodyParam answer_id int required ID wybranej odpowiedzi. Example: 12
     *
     * @response 200 scenario="Odpowiedź zapisana poprawnie" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "Answer has been saved"
     * }
     * @response 404 scenario="Brak pytania lub już udzielono odpowiedzi" {
     *   "status": "ERROR",
     *   "message": "I couldn't find a question associated with this duel"
     * }
     * @response 409 scenario="Odpowiedź już udzielona" {
     *   "status": "ERROR",
     *   "message": "The answer to this question has already been given"
     * }
     * @response 422 scenario="Za późno na odpowiedź" {
     *   "status": "ERROR",
     *   "message": "The answer to this question was given too late"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function __invoke(Duel $duel, DuelAnswerRequest $request): JsonResponse
    {
        $userId = (int) auth()->id();
        $questionId = (int) $request->input('question_id');
        $answerId = (int) $request->input('answer_id');

        try {
            $this->answerDuelQuestion->handle($duel, $userId, $questionId, $answerId);
        } catch (DuelQuestionNotFoundException $e) {
            return Response::error(message: $e->getMessage(), status: 404);
        } catch (AnswerAlreadyGivenException $e) {
            return Response::error(message: $e->getMessage(), status: 409);
        } catch (AnswerTimeoutException $e) {
            return Response::error(message: $e->getMessage(), status: 422);
        } catch (Exception $e){
            report($e);
            return Response::generalError();
        }

        return Response::success(message: 'Answer has been saved');
    }
}
