<?php

namespace Albedo\Duel\Filament\Resources\Duels\Tables;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultStatusEnum;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUser;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;

class DuelsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('duelRound.name')
                    ->searchable()
                    ->description(fn(Duel $duel) => "{$duel->duelRound->begin_date->format('d.m.Y')}-{$duel->duelRound->end_date->format('d.m.Y')}")
                    ->label('Round'),
                TextColumn::make('duelUsers')
                    ->badge()
                    ->formatStateUsing(fn(DuelUser $state) => $state->user->full_name)
                    ->label('Participants')
                    ->icon(fn(DuelUser $state) => match ($state->result_status) {
                        DuelResultStatusEnum::WINNER => Heroicon::OutlinedTrophy,
                        DuelResultStatusEnum::DRAW => Heroicon::OutlinedArrowsRightLeft,
                        default => null
                    })
                    ->color(fn(DuelUser $state) => match ($state->result_status) {
                        DuelResultStatusEnum::WINNER => 'success',
                        DuelResultStatusEnum::LOSER => 'danger',
                        DuelResultStatusEnum::UNKNOWN => 'gray',
                        DuelResultStatusEnum::DRAW => 'info',
                        default => null,
                    })
                    ->separator(', '),
                TextColumn::make('progress_status')
                    ->icon(fn($state) => match ($state) {
                        DuelProgressStatusEnum::NOT_PARTICIPATED => Heroicon::OutlinedNoSymbol,
                        DuelProgressStatusEnum::IN_PROGRESS      => Heroicon::OutlinedClock,
                        DuelProgressStatusEnum::FINISHED         => Heroicon::OutlinedCheckCircle,
                        default => null,
                    })
                    ->color(fn($state) => match ($state) {
                        DuelProgressStatusEnum::NOT_PARTICIPATED => 'gray',
                        DuelProgressStatusEnum::IN_PROGRESS => 'info',
                        DuelProgressStatusEnum::FINISHED => 'success',
                        default => 'secondary',
                    })
                    ->iconColor(fn($state) => match ($state) {
                        DuelProgressStatusEnum::NOT_PARTICIPATED => 'gray',
                        DuelProgressStatusEnum::IN_PROGRESS => 'info',
                        DuelProgressStatusEnum::FINISHED => 'success',
                        default => 'secondary',
                    })
                    ->label('Status'),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('deleted_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TrashedFilter::make(),
            ])
            ->recordActions([
                ViewAction::make()
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ]);
    }
}
