<?php

namespace Albedo\Duel\Services;

use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Enums\DuelResultStatusEnum;
use Albedo\Duel\Events\DuelResolvedEvent;
use Albedo\Duel\Events\UserWonDuelEvent;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUser;
use Albedo\Duel\Settings\DuelSettings;
use App\Models\User;
use Illuminate\Support\Collection;

class DuelService
{

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getNonCompletedDuels(): \Illuminate\Database\Eloquent\Collection
    {
        return Duel::whereIn('progress_status', [
            DuelProgressStatusEnum::IN_PROGRESS,
            DuelProgressStatusEnum::NOT_PARTICIPATED
        ])
            ->get();
    }

    public static function getMaxTotalAnswerTimeInMs(DuelSettings $duelSettings): int
    {
        $maxAnswerTime = $duelSettings->answer_time_absolute_in_seconds;
        $answerTimeBuffer = $duelSettings->answer_time_buffer_in_seconds;

        return ($maxAnswerTime + $answerTimeBuffer) * 1000;
    }

    public function resolveDuel(Duel $duel): void
    {
        $rankedDuelUsers = $this->getRankedDuelUsers($duel);

        $results = $this->categorizeResults($rankedDuelUsers);

        $this->updateDuelUsersStatus($results);
        $this->updateDuelStatus($duel);
        $this->dispatchDuelEvents($results);
    }

    private function getRankedDuelUsers(Duel $duel): Collection
    {
        return $duel->duelUsers()
            ->orderBy('points', 'desc')
            ->orderBy('time', 'asc')
            ->orderBy('started_at', 'asc')
            ->get();
    }

    private function categorizeResults(Collection $rankedDuelUsers): array
    {
        $noOneParticipated = $rankedDuelUsers->every(fn(DuelUser $duelUser) => $duelUser->notParticipated());

        if ($noOneParticipated) {
            return [
                'winners' => [],
                'losers' => $rankedDuelUsers->all(),
            ];
        }

        $participants = $rankedDuelUsers->filter(fn(DuelUser $duelUser) => $duelUser->hasParticipated());

        if ($participants->count() === 1) {
            $nonParticipants = $rankedDuelUsers->filter(fn(DuelUser $duelUser) => $duelUser->notParticipated());

            return [
                'winners' => [$participants->first()],
                'losers' => $nonParticipants->all(),
            ];
        }

        $winner = $rankedDuelUsers->first();
        $losers = $rankedDuelUsers->slice(1)->all();

        return [
            'winners' => [$winner],
            'losers' => $losers,
        ];
    }

    private function updateDuelUsersStatus(array $results): void
    {
        foreach ($results['winners'] as $winner) {
            $winner->update([
                'result_status' => DuelResultStatusEnum::WINNER
            ]);
        }

        foreach ($results['losers'] as $loser) {
            $loser->update([
                'result_status' => DuelResultStatusEnum::LOSER
            ]);
        }
    }

    private function updateDuelStatus(Duel $duel): void
    {
        $duel->update([
            'progress_status' => DuelProgressStatusEnum::FINISHED
        ]);

        $duel->duelUsers()->update([
            'progress_status' => DuelProgressStatusEnum::FINISHED
        ]);
    }

    private function dispatchDuelEvents(array $results): void
    {
        foreach ($results['winners'] as $winner) {
            event(new UserWonDuelEvent($winner));
            event(new DuelResolvedEvent($winner));
        }

        foreach ($results['losers'] as $loser) {
            event(new DuelResolvedEvent($loser));
        }
    }
}
