<?php

namespace Albedo\Duel\Http\Controllers;


use Albedo\Duel\Events\DuelCorrectAnswerGivenEvent;
use Albedo\Duel\Events\DuelPossibleEndEvent;
use Albedo\Duel\Events\DuelQuestionAnsweredEvent;
use Albedo\Duel\Http\Requests\DuelAnswerRequest;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelUserAnswer;
use Albedo\Duel\Services\DuelService;
use Albedo\Duel\Settings\DuelSettings;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

class DuelAnswerAction extends Controller
{
    public function __construct(
        protected DuelService $duelService,
        protected DuelSettings $duelSettings
    ) {}

    /**
     * Odpowiedź pojedynkowa
     *
     * Zapisuje odpowiedź użytkownika na konkretne pytanie w ramach wskazanego pojedynku.
     *
     * @group Pojedynki
     *
     * @urlParam duel int required ID pojedynku. Example: 123
     * @bodyParam question_id int required ID pytania. Example: 5
     * @bodyParam answer_id int required ID wybranej odpowiedzi. Example: 12
     *
     * @response 200 scenario="Odpowiedź zapisana poprawnie" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "Answer has been saved"
     * }
     * @response 404 scenario="Brak pytania lub już udzielono odpowiedzi" {
     *   "status": "ERROR",
     *   "message": "I couldn't find a question associated with this duel"
     * }
     * @response 409 scenario="Odpowiedź już udzielona" {
     *   "status": "ERROR",
     *   "message": "The answer to this question has already been given"
     * }
     * @response 422 scenario="Za późno na odpowiedź" {
     *   "status": "ERROR",
     *   "message": "The answer to this question was given too late"
     * }
     * @response 403 { "message": "Forbidden" }
     */
    public function __invoke(Duel $duel, DuelAnswerRequest $request): JsonResponse
    {
        $userId = auth()->id();
        $now = now();

        $questionId = $request->input('question_id');
        $answerId = $request->input('answer_id');

        $duelUserAnswer = DuelUserAnswer::query()
            ->lockForUpdate()
            ->whereDuelId($duel->id)
            ->whereRelation('duelUser', 'user_id', $userId)
            ->whereDuelQuestionId($questionId)
            ->whereNotNull('answer_expiration_date')
            ->orderBy('number')
            ->first();

        if (!$duelUserAnswer) {
            return Response::error(message: 'I couldn\'t find a question associated with this duel', status: 404);
        }

        if ($duelUserAnswer->duel_answer_id) {
            return Response::error(message: 'The answer to this question has already been given', status: 409);
        }

        $maxAnswerTime = $this->duelSettings->answer_time_absolute_in_seconds * 1000;
        $maxTotalTime = DuelService::getMaxTotalAnswerTimeInMs($this->duelSettings);

        if ($duelUserAnswer->answer_expiration_date <= $now) {
            $duelUserAnswer->answer_time = $maxTotalTime - $now->diffInMilliseconds($duelUserAnswer->answer_expiration_date);
            $duelUserAnswer->save();
            return Response::error(message: 'The answer to this question was given too late', status: 422);
        }

        $duelUserAnswer->duel_answer_id = $answerId;
        $duelUserAnswer->answer_time = $maxTotalTime - $now->diffInMilliseconds($duelUserAnswer->answer_expiration_date);

        if ($duelUserAnswer->answer_time > $maxAnswerTime) {
            $duelUserAnswer->answer_time = $maxAnswerTime;
        }

        $duelUserAnswer->save();

        $duelUserAnswer = $duelUserAnswer->fresh();

        $duelUser = $duelUserAnswer->duelUser;
        if ($duelUserAnswer->duelAnswer?->is_valid && $duelUserAnswer->duelAnswer?->duel_question_id == $duelUserAnswer->duel_question_id) {
            $duelUser->points += $this->duelSettings->points_for_correct_answer;
            $duelUserAnswer->is_correct = true;
            $duelUserAnswer->save();
            event(new DuelCorrectAnswerGivenEvent($duelUserAnswer));
        }else {
            $duelUserAnswer->is_correct = false;
            $duelUserAnswer->save();
        }

        $duelUser->ended_at = $now;
        $duelUser->save();

        event(new DuelQuestionAnsweredEvent($duelUserAnswer));
        event(new DuelPossibleEndEvent($duel));

        return Response::success(message: 'Answer has been saved');
    }
}
