<?php

namespace Albedo\Duel\Http\Controllers;


use Albedo\Duel\Enums\DuelProgressStatusEnum;
use Albedo\Duel\Http\Resources\DuelResource;
use Albedo\Duel\Models\Duel;
use Albedo\Duel\Models\DuelRound;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;


class DuelCurrentAction extends Controller
{

    /**
     * Aktualny pojedynek
     *
     * Endpoint zwraca aktualny pojedynek zalogowanego użytkownika w aktywnej rundzie pojedynków.
     * Jeśli nie ma aktywnej rundy lub użytkownik nie bierze udziału w żadnym pojedynku, zwracany jest pusty obiekt.
     *
     * @group Pojedynki
     *
     * @response 200 scenario="Użytkownik ma aktywny pojedynek" {
     *   "status": "OK",
     *   "data": {
     *     "id": 1,
     *     "progress_status": "in_progress",
     *     "me": {
     *       "user_id": 10,
     *       "progress_status": "in_progress",
     *       "answered": [ ... ]
     *     },
     *     "opponent": {
     *       "user_id": 11,
     *       "progress_status": "not_participated",
     *       "answered": [ ... ]
     *     }
     *   },
     *   "message": "Success"
     * }
     * @response 200 scenario="Brak pojedynku dla użytkownika" {
     *   "status": "OK",
     *   "data": [],
     *   "message": "Success"
     * }
     * @response 404 scenario="Brak aktywnej rundy" {
     *   "status": "ERROR",
     *   "message": "No active duel round found"
     * }
     */
    public function __invoke()
    {
        $duelRound = DuelRound::getCurrent();

        if (!$duelRound) {
            return Response::error(message: 'No active duel round found', status: 404);
        }

        $userId = auth()->id();

        $duel = Duel::whereHas('duelUsers', fn($q) => $q
            ->where('user_id', $userId)
            ->where('duel_round_id', $duelRound->id)
            ->whereIn('progress_status', [DuelProgressStatusEnum::IN_PROGRESS, DuelProgressStatusEnum::NOT_PARTICIPATED])
        )->with([
            'duelUsers.duelUserAnswers',
            'duelUsers.duelUserAnswers.duelQuestion.duelQuestionCategory'
        ])
            ->first();

        if (!$duel) {
            return Response::error(message: 'No active duel found', status: 404);
        }

        return Response::success(DuelResource::make($duel)->additional([
            'me' => $duel->duelUsers->where('user_id', auth()->id())->first(),
            'opponent' => $duel->duelUsers->where('user_id', '!=', auth()->id())->first(),
        ]));
    }
}
