<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Http\Requests\UserResetPasswordRequest;
use Albedo\Auth\Services\ActionCodeService;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;

class ResetPasswordAction extends Controller
{
    public function __construct(protected ActionCodeService $actionCodeService)
    {

    }
    public function __invoke(UserResetPasswordRequest $request): JsonResponse
    {
        $record = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$record || ! Hash::check($request->token, $record->token)) {
            return Response::error(message: 'Invalid token');
        }

        if (Carbon::parse($record->created_at)->addSeconds(config('albedo-auth.password_reset_token_expires_in'))->isPast()) {
            return Response::error(message: 'Token expired');
        }

        $user = app(config('albedo-auth.user_model'))::query()
            ->whereEmail($request->email)
            ->first();

        if (is_null($user)) {
            return Response::error(
                message: 'User has not been found.'
            );
        }

        $user->forceFill([
            'password' => Hash::make($request->password)
        ])
            ->setRememberToken(Str::random(60));

        $user->save();

        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        event(new PasswordReset($user));

        return Response::success(
            message: __('Password has been changed.'),
        );


    }
}
