<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Contracts\LogsInParticipantsInterface;
use Albedo\Auth\Exceptions\IncorrectLoginException;
use Albedo\Auth\Exceptions\UnverifiedAccountException;
use Albedo\Auth\Http\Requests\LoginRequest;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * Akcja logowania użytkownika.
 *
 * @group Autoryzacja
 * Endpoint umożliwiający zalogowanie się użytkownika.
 *
 * @bodyParam login string required Email użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam password string required Hasło użytkownika. Example: SilneHasło123!
 *
 * @response 200 {"message": "Login successful.", "data": {"token": "..."}}
 * @response 401 {"message": "Incorrect email or password..."}
 * @response 403 {"message": "Account has not been verified."}
 */
class LoginAction extends Controller
{

    public function __construct(protected LogsInParticipantsInterface $authService)
    {
    }

    /**
     * Loguje użytkownika na podstawie przekazanych danych.
     *
     * @param LoginRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function __invoke(LoginRequest $request)
    {
        try {
            $token = $this->authService->login($request->validated(), $this->getClient($request));
        } catch (IncorrectLoginException|UnverifiedAccountException $ex) {
            return Response::error(
                message: $ex->getMessage(),
                status: $ex->getCode()
            );
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }

        return Response::success(
            data: [
                'token' => $token,
            ],
            message: 'Login successful.'
        );
    }

    /**
     * @param Request $request
     * @return array|string|null
     */
    public function getClient(Request $request): string|array|null
    {
        return $request->header('X-Client-Type', 'web');
    }
}
