<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Contracts\RecoversParticipantPasswordsInterface;
use Albedo\Auth\Http\Requests\RemindPasswordRequest;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * Wysłanie kodu do resetu hasła.
 *
 * @group Reset hasła
 * Endpoint do żądania **kodu** do zresetowania hasła przez e-mail.
 *
 * @bodyParam email string required Adres e-mail użytkownika. Example: j.kowalski@domena.pl
 * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
 */
class RemindPasswordAction extends Controller
{
    public function __construct(protected RecoversParticipantPasswordsInterface $authService)
    {
    }

    /**
     * Wysyła kod resetujący hasło na podany e-mail jeśli użytkownik istnieje.
     *
     * @param RemindPasswordRequest $request
     * @return JsonResponse
     * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
     */
    public function __invoke(RemindPasswordRequest $request)
    {
        try {
            $this->authService->remind($request->validated());
        } catch (ModelNotFoundException $ex) {
            return $this->responseSuccess();
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }

        return $this->responseSuccess();
    }

    /**
     * Sukces dla prośby o reset (niezależnie czy mail istnieje).
     *
     * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
     */
    public function responseSuccess(): \Illuminate\Http\JsonResponse
    {
        return Response::success(
            message: __('If the provided email address exists in our system, we will send instructions to reset your password.')
        );
    }
}
