<?php

namespace Albedo\Auth\Services;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Models\ActionCode;
use Illuminate\Contracts\Auth\Authenticatable;

class ActionCodeService
{
    public const TYPE_VERIFY = 'verify';
    public const TYPE_RESET = 'reset';

    public function generate(Authenticatable $user, ActionType $type, int $ttlMinutes = 10): ActionCode
    {
        $code = str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);

        // Invalidate previous unused codes of the same type
        ActionCode::query()
            ->where('user_id', $user->getAuthIdentifier())
            ->where('type', $type)
            ->whereNull('consumed_at')
            ->update(['consumed_at' => now()]);

        return ActionCode::query()->create([
            'user_id' => $user->getAuthIdentifier(),
            'type' => $type,
            'code' => $code,
            'expires_at' => now()->addMinutes($ttlMinutes),
        ]);
    }

    public function validateAndConsume(Authenticatable $user, ActionType $type, string $code): bool
    {
        $actionCode = ActionCode::query()
            ->where('user_id', $user->getAuthIdentifier())
            ->where('type', $type)
            ->where('code', $code)
            ->latest('id')
            ->first();

        if (!$actionCode) {
            return false;
        }

        if ($actionCode->isExpired() || $actionCode->isConsumed()) {
            return false;
        }

        $actionCode->markConsumed();
        return true;
    }
}


