<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Http\Requests\LoginRequest;
use Illuminate\Auth\Events\Login;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;

/**
 * Akcja logowania użytkownika.
 *
 * @group Autoryzacja
 * Endpoint umożliwiający zalogowanie się użytkownika.
 *
 * @bodyParam login string required Email użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam password string required Hasło użytkownika. Example: SilneHasło123!
 *
 * @response 200 {"message": "Login successful.", "data": {"token": "..."}}
 * @response 401 {"message": "Incorrect email or password..."}
 * @response 403 {"message": "Account has not been verified."}
 */
class LoginAction extends Controller
{
    /**
     * Loguje użytkownika na podstawie przekazanych danych.
     *
     * @param LoginRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function __invoke(LoginRequest $request)
    {
        $login = $request->login;
        $password = $request->password;

        $user = app(config('albedo-auth.user_model'))::query()
            ->where('email', $login)
            ->first();

        if (is_null($user) || !Hash::check($password, $user->password)) {
            return Response::error(message: __('Incorrect email or password...'), status: 401);
        }

        if (!$user->hasVerifiedEmail()) {
            return Response::error(message: __('Account has not been verified.'), status: 403);
        }

        $token = $user->createToken('api')->plainTextToken;

        event(new Login('api', $user, false));

        return Response::success(
            message: 'Login successful.',
            data: [
                'token' => $token,
            ]);
    }
}
