<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Http\Requests\VerificationCodeResetPasswordRequest;
use Albedo\Auth\Services\ActionCodeService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;

/**
 * Uzyskanie tokenu resetującego hasło na podstawie kodu z e-maila.
 *
 * @group Reset hasła
 * Endpoint pozwala na wygenerowanie tokenu do zmiany hasła na podstawie poprawnego **kodu** przesłanego na adres e-mail.
 *
 * @bodyParam email string required E-mail użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam code string required Kod z e-maila. Example: 984563
 *
 * @response 200 {"data": {"token": "..."}, "message": "Password reset token generated successfully."}
 * @response 422 {"message": "Invalid or expired code."}
 * @response 422 {"message": "User has not been found."}
 */
class VerificationCodeResetPasswordAction extends Controller
{

    public function __construct(protected ActionCodeService $actionCodeService)
    {

    }

    /**
     * Generuje token resetu hasła na podstawie poprawnego kodu z maila.
     *
     * @param VerificationCodeResetPasswordRequest $request
     *
     * @return JsonResponse
     * @response 200 {"data": {"token": "..."}, "message": "Password reset token generated successfully."}
     * @response 422 {"message": "Invalid or expired code."}
     * @response 422 {"message": "User has not been found."}
     */
    public function __invoke(VerificationCodeResetPasswordRequest $request)
    {
        $user = app(config('albedo-auth.user_model'))::query()
            ->whereEmail($request->email)
            ->first();

        if (is_null($user)) {
            return Response::error(
                message: 'User has not been found.'
            );
        }

        $code = $request->input('code');

        if (!$this->actionCodeService->validateAndConsume($user, ActionType::RESET_PASSWORD, $code)) {
            return Response::error(
                message: __('Invalid or expired code.'), status: 422
            );
        }

        $token = Str::random(64);

        DB::table('password_reset_tokens')->where('email', $user->email)->delete();

        DB::table('password_reset_tokens')->insert([
            'email' => $user->email,
            'token' => Hash::make($token),
            'created_at' => now(),
        ]);

        return Response::success(
            data: ['token' => $token],
            message: __('Password reset token generated successfully.'),
        );
    }
}
