<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Http\Requests\UserResetPasswordRequest;
use Albedo\Auth\Services\ActionCodeService;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;

/**
 * Ustawienie nowego hasła na podstawie kodu resetującego.
 *
 * @group Reset hasła
 * Endpoint pozwala na ustawienie nowego hasła po poprawnej weryfikacji **kodu resetującego**.
 *
 * @bodyParam email string required E-mail użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam token string required Token resetujący hasło.
 * @bodyParam password string required Nowe hasło. Example: SilneHasło123!
 * @bodyParam password_confirmation string required Potwierdzenie nowego hasła. Example: SilneHasło123!
 *
 * @response 200 {"message": "Password has been changed."}
 * @response 422 {"message": "Invalid token"}
 * @response 422 {"message": "Token expired"}
 * @response 422 {"message": "User has not been found."}
 */
class ResetPasswordAction extends Controller
{
    public function __construct(protected ActionCodeService $actionCodeService)
    {

    }
    /**
     * Resetuje hasło użytkownika na podstawie tokenu.
     *
     * @param UserResetPasswordRequest $request
     *
     * @response 200 {"message": "Password has been changed."}
     * @response 422 {"message": "Invalid token"}
     * @response 422 {"message": "Token expired"}
     * @response 422 {"message": "User has not been found."}
     */
    public function __invoke(UserResetPasswordRequest $request): JsonResponse
    {
        $record = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$record || ! Hash::check($request->token, $record->token)) {
            return Response::error(message: 'Invalid token');
        }

        if (Carbon::parse($record->created_at)->addSeconds(config('albedo-auth.password_reset_token_expires_in'))->isPast()) {
            return Response::error(message: 'Token expired');
        }

        $user = app(config('albedo-auth.user_model'))::query()
            ->whereEmail($request->email)
            ->first();

        if (is_null($user)) {
            return Response::error(
                message: 'User has not been found.'
            );
        }

        $user->forceFill([
            'password' => Hash::make($request->password)
        ])
            ->setRememberToken(Str::random(60));

        $user->save();

        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        event(new PasswordReset($user));

        return Response::success(
            message: __('Password has been changed.'),
        );


    }
}
