<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Http\Requests\RemindPasswordRequest;
use Albedo\Auth\Mail\ResetPasswordMail;
use Albedo\Auth\ResetPasswordCodeService;
use Albedo\Auth\Services\ActionCodeService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Response;

/**
 * Wysłanie kodu do resetu hasła.
 *
 * @group Reset hasła
 * Endpoint do żądania **kodu** do zresetowania hasła przez e-mail.
 *
 * @bodyParam email string required Adres e-mail użytkownika. Example: j.kowalski@domena.pl
 * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
 */
class RemindPasswordAction extends Controller
{
    public function __construct(protected ActionCodeService $actionCodeService)
    {
    }

    /**
     * Wysyła kod resetujący hasło na podany e-mail jeśli użytkownik istnieje.
     *
     * @param RemindPasswordRequest $request
     * @return JsonResponse
     * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
     */
    public function __invoke(RemindPasswordRequest $request)
    {
        $email = request()->input('email');

        $user = app(config('albedo-auth.user_model'))::query()
            ->whereEmail($email)
            ->first();

        if (is_null($user)) {
            return $this->responseSuccess();
        }

        $actionCode = $this->actionCodeService->generate($user, ActionType::RESET_PASSWORD);

        Mail::to($user->email)
            ->queue(new ResetPasswordMail($actionCode->code));

        return $this->responseSuccess();
    }

    /**
     * Sukces dla prośby o reset (niezależnie czy mail istnieje).
     *
     * @response 200 {"message": "If the provided email address exists in our system, we will send instructions to reset your password."}
     */
    public function responseSuccess(): \Illuminate\Http\JsonResponse
    {
        return Response::success(
            message: __('If the provided email address exists in our system, we will send instructions to reset your password.')
        );
    }
}
