<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Http\Requests\VerificationRequest;
use Albedo\Auth\Services\ActionCodeService;
use Illuminate\Auth\Events\Verified;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Response;

/**
 * Weryfikacja konta przez kod przesłany e-mailem.
 *
 * @group Weryfikacja konta
 * Endpoint do potwierdzenia adresu e-mail użytkownika kodem z maila.
 *
 * @bodyParam email string required Email użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam code string required Kod weryfikacyjny przesłany e-mailem. Example: 546822
 * @response 200 {"message": "Your account has been verified."}
 * @response 422 {"message": "Invalid or expired code."}
 * @response 422 {"message": "The account has already been verified."}
 */
class VerificationAction extends Controller
{
    public function __construct(protected ActionCodeService $actionCodeService)
    {

    }

    /**
     * Weryfikuje konto użytkownika na podstawie kodu weryfikacyjnego.
     *
     * @param VerificationRequest $request
     *
     * @return JsonResponse
     * @response 200 {"message": "Your account has been verified."}
     * @response 422 {"message": "Invalid or expired code."}
     */
    public function __invoke(VerificationRequest $request)
    {
        $user = app(config('albedo-auth.user_model'))::query()
            ->whereEmail($request->input('email'))
            ->firstOrFail();

        $code = $request->input('code');

        if ($user->hasVerifiedEmail()) {
            return Response::error(
                message: 'The account has already been verified.'
            );
        }

        if (!$this->actionCodeService->validateAndConsume($user, ActionType::VERIFICATION_ACCOUNT, $code)) {
            return Response::error(
                message: 'Invalid or expired code.'
            );
        }

        $user->markEmailAsVerified();

        event(new Verified($user));

        return Response::success(
            message: 'Your account has been verified.',
        );
    }
}
