<?php

namespace Albedo\Auth\Services;

use Albedo\Auth\Contracts\LogsOutParticipantsInterface;
use Albedo\Auth\Contracts\RecoversParticipantPasswordsInterface;
use Albedo\Auth\Enums\ActionType;
use Albedo\Auth\Exceptions\AlreadyUsedPasswordResetTokenException;
use Albedo\Auth\Exceptions\ExpiredPasswordResetTokenException;
use Albedo\Auth\Exceptions\InvalidPasswordResetTokenException;
use Albedo\Auth\Mail\ResetPasswordMail;
use Albedo\Auth\Providers\DefaultAuthUserProvider;
use Albedo\Auth\Settings\CodeExpirationTimeSettings;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class PasswordRecoveryService implements RecoversParticipantPasswordsInterface
{
    public function __construct(
        protected DefaultAuthUserProvider $authUserProvider,
        protected ActionCodeService       $actionCodeService,
        protected LogsOutParticipantsInterface $logoutService,
        protected CodeExpirationTimeSettings   $codeExpirationTimeSettings
    )
    {
    }

    /**
     * @param array $data
     * @return void
     */
    public function remind(array $data): void
    {
        $email = $data['email'];

        $user = $this->getUserBy($email);

        $ttlTime = $this->codeExpirationTimeSettings->password_reset_code;

        $actionCode = $this->actionCodeService->generate($user, ActionType::RESET_PASSWORD, $ttlTime);

        $this->sendResetPasswordMail($user, $actionCode);
    }

    /**
     * @param array $data
     * @return string
     * @throws AlreadyUsedPasswordResetTokenException
     * @throws ExpiredPasswordResetTokenException
     * @throws InvalidPasswordResetTokenException
     */
    public function verify(array $data): string
    {
        $email = $data['email'];
        $code = $data['code'];

        $user = $this->getUserBy($email);

        $actionCode = $this->actionCodeService->getActionCode($user, ActionType::RESET_PASSWORD, $code);

        if (!$actionCode) {
            throw new InvalidPasswordResetTokenException(message: __('Invalid password reset code.'));
        }

        if ($actionCode->isExpired()) {
            throw new ExpiredPasswordResetTokenException(message: __('Expired password reset code.'));
        }

        if ($actionCode->isConsumed()) {
            throw new AlreadyUsedPasswordResetTokenException(message: __('This password reset code has already been used.'));
        }

        $actionCode->markConsumed();

        $token = $this->generateToken();

        $this->createToken($user, $token);

        return $token;
    }

    /**
     * @param array $data
     * @return void
     * @throws ExpiredPasswordResetTokenException
     * @throws InvalidPasswordResetTokenException
     */
    public function reset(array $data): void
    {
        $email = $data['email'];
        $token = $data['token'];
        $password = $data['password'];

        $record = $this->getTokenBy($email);

        if (!$record || !Hash::check($token, $record->token)) {
            throw new InvalidPasswordResetTokenException(message: __('Invalid token.'));
        }

        if (Carbon::parse($record->created_at)->addSeconds(config('albedo-auth.password_reset_token_expires_in'))->isPast()) {
            throw new ExpiredPasswordResetTokenException(message: __('Token has expired.'));
        }

        $user = $this->getUserBy($email);
        $user->forceFill(['password' => Hash::make($password)]);
        $user->save();

        $this->clearPasswordResetTokens($user);

        $this->logoutService->logoutFromAllDevices();

        event(new PasswordReset($user));
    }

    /**
     * @param $user
     * @param \Albedo\Auth\Models\ActionCode $actionCode
     * @return void
     */
    protected function sendResetPasswordMail($user, \Albedo\Auth\Models\ActionCode $actionCode): void
    {
        Mail::to($user->email)
            ->queue(new ResetPasswordMail($actionCode->code));
    }

    /**
     * @param mixed $email
     * @return mixed
     */
    protected function getUserBy(string $email)
    {
        return $this->authUserProvider->getUserModel()::query()
            ->where('email', $email)
            ->firstOrFail();
    }

    /**
     * @param mixed $user
     * @param string $token
     * @return void
     */
    protected function createToken(mixed $user, string $token): void
    {
        DB::table('password_reset_tokens')
            ->where('email', $user->email)
            ->delete();

        DB::table('password_reset_tokens')->insert([
            'email' => $user->email,
            'token' => Hash::make($token),
            'created_at' => now(),
        ]);
    }

    /**
     * @param mixed $email
     * @return object|null
     */
    public function getTokenBy(mixed $email): ?object
    {
        return DB::table('password_reset_tokens')
            ->where('email', $email)
            ->first();
    }

    /**
     * @param mixed $user
     * @return void
     */
    protected function clearPasswordResetTokens(mixed $user): void
    {
        DB::table('password_reset_tokens')->where('email', $user->email)->delete();
    }

    /**
     * @return string
     */
    protected function generateToken(): string
    {
        return Str::random(64);
    }
}
