<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Contracts\VerifiesParticipantsInterface;
use Albedo\Auth\Exceptions\InvalidVerificationException;
use Albedo\Auth\Http\Requests\VerificationRequest;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * Weryfikacja konta przez kod przesłany e-mailem.
 *
 * @group Weryfikacja konta
 * Endpoint do potwierdzenia adresu e-mail użytkownika kodem z maila.
 *
 * @bodyParam email string required Email użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam code string required Kod weryfikacyjny przesłany e-mailem. Example: 546822
 * @response 200 {"message": "Your account has been verified."}
 * @response 422 {"message": "Invalid or expired code."}
 * @response 422 {"message": "The account has already been verified."}
 */
class VerificationAction extends Controller
{
    public function __construct(protected VerifiesParticipantsInterface $authService)
    {

    }

    /**
     * Weryfikuje konto użytkownika na podstawie kodu weryfikacyjnego.
     *
     * @param VerificationRequest $request
     *
     * @return JsonResponse
     * @response 200 {"message": "Your account has been verified."}
     * @response 422 {"message": "Invalid or expired code."}
     */
    public function __invoke(VerificationRequest $request)
    {
        try {
            $this->authService->verify($request->validated());
        } catch (InvalidVerificationException $e) {
            return Response::error(
                message: $e->getMessage(),
                status: $e->getCode(),
            );
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }

        return Response::success(
            message: 'Your account has been verified.',
        );
    }
}
