<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Contracts\RecoversParticipantPasswordsInterface;
use Albedo\Auth\Exceptions\AlreadyUsedPasswordResetTokenException;
use Albedo\Auth\Exceptions\ExpiredPasswordResetTokenException;
use Albedo\Auth\Exceptions\InvalidPasswordResetTokenException;
use Albedo\Auth\Http\Requests\VerificationCodeResetPasswordRequest;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * Uzyskanie tokenu resetującego hasło na podstawie kodu z e-maila.
 *
 * @group Reset hasła
 * Endpoint pozwala na wygenerowanie tokenu do zmiany hasła na podstawie poprawnego **kodu** przesłanego na adres e-mail.
 *
 * @bodyParam email string required E-mail użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam code string required Kod z e-maila. Example: 984563
 *
 * @response 200 {"data": {"token": "..."}, "message": "Password reset token generated successfully."}
 * @response 422 {"message": "Invalid or expired code."}
 * @response 422 {"message": "User has not been found."}
 */
class VerificationCodeResetPasswordAction extends Controller
{

    public function __construct(protected RecoversParticipantPasswordsInterface $authService)
    {

    }

    /**
     * Generuje token resetu hasła na podstawie poprawnego kodu z maila.
     *
     * @param VerificationCodeResetPasswordRequest $request
     *
     * @return JsonResponse
     * @response 200 {"data": {"token": "..."}, "message": "Password reset token generated successfully."}
     * @response 422 {"message": "Invalid or expired code."}
     * @response 422 {"message": "User has not been found."}
     */
    public function __invoke(VerificationCodeResetPasswordRequest $request)
    {
        try {
            $token = $this->authService->verify($request->validated());
        } catch (ExpiredPasswordResetTokenException|AlreadyUsedPasswordResetTokenException|InvalidPasswordResetTokenException $e) {
            return Response::error(
                message: $e->getMessage(),
                status: $e->getCode()
            );
        } catch (ModelNotFoundException $e) {
            return Response::error(
                message: __('Invalid password reset code.'),
                status: 409
            );
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }

        return Response::success(
            data: ['token' => $token],
            message: __('Password reset token generated successfully.'),
        );
    }
}
