<?php

namespace Albedo\Auth\Http\Controllers;

use Albedo\Auth\Contracts\RecoversParticipantPasswordsInterface;
use Albedo\Auth\Exceptions\ExpiredPasswordResetTokenException;
use Albedo\Auth\Exceptions\InvalidPasswordResetTokenException;
use Albedo\Auth\Http\Requests\UserResetPasswordRequest;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;

/**
 * Ustawienie nowego hasła na podstawie kodu resetującego.
 *
 * @group Reset hasła
 * Endpoint pozwala na ustawienie nowego hasła po poprawnej weryfikacji **kodu resetującego**.
 *
 * @bodyParam email string required E-mail użytkownika. Example: j.kowalski@domena.pl
 * @bodyParam token string required Token resetujący hasło.
 * @bodyParam password string required Nowe hasło. Example: SilneHasło123!
 * @bodyParam password_confirmation string required Potwierdzenie nowego hasła. Example: SilneHasło123!
 *
 * @response 200 {"message": "Password has been changed."}
 * @response 422 {"message": "Invalid token"}
 * @response 422 {"message": "Token expired"}
 * @response 422 {"message": "User has not been found."}
 */
class ResetPasswordAction extends Controller
{
    public function __construct(protected RecoversParticipantPasswordsInterface $authService)
    {
    }

    /**
     * Resetuje hasło użytkownika na podstawie tokenu.
     *
     * @param UserResetPasswordRequest $request
     *
     * @response 200 {"message": "Password has been changed."}
     * @response 422 {"message": "Invalid token"}
     * @response 422 {"message": "Token expired"}
     * @response 422 {"message": "User has not been found."}
     */
    public function __invoke(UserResetPasswordRequest $request): JsonResponse
    {
        try {
            $this->authService->reset($request->validated());
        } catch (InvalidPasswordResetTokenException|ExpiredPasswordResetTokenException $e) {
            return Response::error(
                message: $e->getMessage(),
                status: $e->getCode()
            );
        } catch (ModelNotFoundException $e) {
            return Response::error(
                message: __('Invalid password reset code.'),
                status: 409
            );
        } catch (Exception $e) {
            report($e);
            return Response::generalError();
        }

        return Response::success(
            message: __('Password has been changed.'),
        );

    }
}
