<?php

namespace Albedo\Auth\Services;

use Albedo\Auth\Contracts\RegistersParticipantsInterface;
use Albedo\Auth\Exceptions\IncorrectLoginException;
use Albedo\Auth\Exceptions\VerifiedAccountException;
use Albedo\Auth\Providers\DefaultAuthUserProvider;
use Albedo\Gamification\Contracts\HasAvatar;
use Albedo\Gamification\Contracts\MustAcceptUserAgreements;
use Albedo\Gamification\Services\AgreementService;
use Albedo\Gamification\Services\AvatarService;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Hash;

class RegisterService implements RegistersParticipantsInterface
{
    public function __construct(
        protected DefaultAuthUserProvider $authUserProvider,
        protected AgreementService        $agreementService,
        protected AvatarService           $avatarService,
    )
    {

    }

    /**
     * @throws VerifiedAccountException
     * @throws IncorrectLoginException
     */
    public function register(array $data)
    {
        $user = $this->findExistingUserForRegistration($data);

        if (is_null($user)){
            $user = $this->authUserProvider->getUserModel()::create($data);
        }

        $this->acceptAgreements($user, $data);

        $this->assignAvatar($user, $data);

        event(new Registered($user));

        return $user;
    }

    /**
     * @param $user
     * @param array $data
     * @return void
     */
    public function acceptAgreements($user, array $data): void
    {
        if ($user instanceof MustAcceptUserAgreements) {
            $this->agreementService->acceptAgreements($user, $data);
        }
    }

    /**
     * @param $user
     * @param array $data
     * @return void
     */
    public function assignAvatar($user, array $data): void
    {
        if ($user instanceof HasAvatar) {
            $this->avatarService->assignAvatarFromRequest($user, $data);
        }
    }

    /**
     * @throws IncorrectLoginException
     * @throws VerifiedAccountException
     */
    public function findExistingUserForRegistration(array $data)
    {
        $email = $data['email'];
        $password = $data['password'];

        $user = $this->authUserProvider->getUserModel()
            ->where('email', $email)
            ->first();

        if (!$user) {
            return null;
        }

        if (!Hash::check($password, $user->password)) {
            throw new IncorrectLoginException();
        }

        if ($user->hasVerifiedEmail()) {
            throw new VerifiedAccountException();
        }

        return $user;
    }
}
